import logging
import os

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox.projects.common.geosearch.personalization_params import YtParams
from sandbox.projects import resource_types
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_logs_folder


class UploaderParams(object):
    ''' Uploader parameters '''

    class SaasIndexer(sdk_parameters.SandboxStringParameter):
        name = 'saas_indexer'
        description = 'SaaS indexer'
        choices = [
            ('testing', 'saas-indexerproxy-maps-prestable.yandex.net'),
            ('stable', 'saas-indexerproxy-maps-kv.yandex.net'),
        ]
        default_value = 'saas-indexerproxy-maps-prestable.yandex.net'

    class SaasDumper(sdk_parameters.ResourceSelector):
        name = 'saas_dumper'
        description = 'SaaS dumper: '
        resource_type = resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE

    class SaasKey(sdk_parameters.SandboxStringParameter):
        name = 'saas_key'
        description = 'key for saas indexer'
        required = True

    class SourceTablePath(sdk_parameters.SandboxStringParameter):
        name = 'source_table_path'
        description = 'Path to source table'
        required = True

    class ErrorsTable(sdk_parameters.SandboxStringParameter):
        name = 'errors_table'
        description = 'Path where to write errors'

    class SaasAction(sdk_parameters.SandboxStringParameter):
        name = 'saas_action'
        description = 'Action for dumper: atAdd/atModify/atDelete/atUpdate/atReopen/atSwitchPrefix'
        required = True
        default_value = 'atModify'

    class MRJobsCount(sdk_parameters.SandboxStringParameter):
        name = 'mr_jobs_count'
        description = 'MR jobs count'

    class SaasProcessor(sdk_parameters.SandboxStringParameter):
        name = 'saas_processor'
        description = 'Saas mr-processor (available processors here: saas/api/mr_client/processors/processors.cpp)'
        required = True

    class Prefix(sdk_parameters.SandboxStringParameter):
        name = 'prefix'
        description = 'Transforms to --prefix parameter of dump_to_indexerproxy. Aka sgkps'

    class MRProcessorEnv(sdk_parameters.SandboxStringParameter):
        name = 'mr_processor_env'
        description = 'Transforms to --mr-processor-env parameter of dump_to_indexerproxy'

    class Deadline(sdk_parameters.SandboxStringParameter):
        name = 'deadline'
        description = '''
            Transforms to --deadline parameter of dump_to_indexerproxy.
            After this point (unixtime in minutes) document will be removed(UTC)
        '''

    Content = [
        SaasIndexer,
        SaasDumper,
        SaasKey,
        SourceTablePath,
        ErrorsTable,
        SaasAction,
        MRJobsCount,
        SaasProcessor,
        Prefix,
        MRProcessorEnv,
        Deadline,
    ]


class Uploader(sdk_task.SandboxTask, UploaderParams, YtParams):
    ''' Runner tool for Dump indexer proxy '''

    CTX_REDEFINES = {
        'kill_timeout': 7 * 60 * 60
    }

    type = 'SAAS_KV_UPLOADER'
    input_parameters = UploaderParams.Content + \
        YtParams.Content

    environment = (sdk_environments.PipEnvironment('yandex-yt'), sdk_environments.PipEnvironment('saaspy'),)

    def initCtx(self):
        self.ctx.update(self.CTX_REDEFINES)

    def on_execute(self):
        indexer = self.ctx.get(self.SaasIndexer.name)
        dumper = self.sync_resource(self.ctx.get(self.SaasDumper.name))
        yt_proxy = self.ctx.get(self.YtProxy.name)
        yt_token_secret_name = self.ctx.get(self.YtTokenSecretName.name)
        yt_token = self.get_vault_data(self.owner, yt_token_secret_name)
        input_table = self.ctx.get(self.SourceTablePath.name)
        errors_table = self.ctx.get(self.ErrorsTable.name)
        saas_key = self.ctx.get(self.SaasKey.name)
        saas_action = self.ctx.get(self.SaasAction.name)
        saas_processor = self.ctx.get(self.SaasProcessor.name)
        mr_jobs_count = self.ctx.get(self.MRJobsCount.name)
        prefix = self.ctx.get(self.Prefix.name)
        mr_processor_env = self.ctx.get(self.MRProcessorEnv.name)
        deadline = self.ctx.get(self.Deadline.name)

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token
        os.environ['YT_LOG_LEVEL'] = 'DEBUG'

        command = [dumper,
                   '--mode', 'mr',
                   '-k', saas_key,
                   '-h', indexer,
                   '--mr-server', yt_proxy,
                   '--mr-input', input_table,
                   '--mr-processor', saas_processor,
                   '--action', saas_action,
                   ]
        if mr_jobs_count:
            command += ['--mr-jobs', mr_jobs_count]
        if errors_table:
            command += ['--mr-output', errors_table]
        if prefix:
            command += ['--prefix', prefix]
        if mr_processor_env:
            command += ['--mr-processor-env', mr_processor_env]
        if deadline:
            command += ['--deadline', deadline]

        logging.info('Run command: {}'.format(' '.join(command)))
        run_process(command, stderr=open(os.path.join(get_logs_folder(), 'dump_to_indexerproxy.stderr'), 'w'))
        logging.info('Done')


__Task__ = Uploader
