# -*- coding: utf-8 -*-
# Валидатор и парсер blockstat-лога.
# Формат записи blockstat-лога:
# path_1 \t vars_count \t var_1=val_1 \t var_2=val_2 \t ... \t path_2 \t vars_count \t ... \t

from collections import deque


class BlocksValidationError(Exception):
    pass


class Block:
    def __init__(self, path, vars_dict):
        """
        :param path:
        :type path: str
        :param vars_dict:
        :type vars_dict: list<str>
        """
        self.path = path
        self.vars = vars_dict

    def __eq__(self, other):
        return self.path == other.path and self.vars == other.vars


def parse_var(var_str):
    """
    :param var_str:
    :type var_str: str
    :return:
    :rtype: (str, str)
    """
    try:
        (key, value) = var_str.split('=', 1)
    except ValueError:
        raise BlocksValidationError('Var {} does not match var pattern'.format(var_str))

    if not key or not value:
        raise BlocksValidationError('Var {} does not match var pattern'.format(var_str))

    return key, value


def parse_block(tokens):
    """
    :param tokens:
    :type tokens: deque<str>
    :return:
    :rtype: Block
    """
    path = tokens.popleft()

    if not tokens:
        raise BlocksValidationError('Expected number of vars but got end of data')

    vars_count_str = tokens.popleft()
    try:
        vars_count = int(vars_count_str)
    except ValueError as e:
        raise BlocksValidationError('Could not parse number of vars from {}: {}'.format(vars_count_str, e))

    if vars_count < 0:
        raise BlocksValidationError('Number of vars should 0 or more, but it is {}'.format(vars_count))

    vars_dict = {}
    for i in xrange(vars_count):
        if not tokens:
            raise BlocksValidationError('Expected var but got end of data')

        (key, value) = parse_var(tokens.popleft())
        vars_dict[key] = value

    return Block(path, vars_dict)


def parse_blocks(data):
    """
    :param data: исходные данные
    :type data: str
    :return: list<BlockstatRecord>
    """
    if not data:
        raise BlocksValidationError('"Blocks" field is empty')

    tokens = deque(data.split('\t'))
    last = tokens.pop()
    if last:
        raise BlocksValidationError('"Blocks" field should end with tab, but it ends with "{}"'.format(last[-1]))

    blocks = []
    while tokens:
        blocks.append(parse_block(tokens))

    return blocks
