import os
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.projects import resource_types
from sandbox.projects.common.userdata import userdata_base_task, util


class UserBrowsePackage(sdk_parameters.LastReleasedResource):
    name = 'user_browse_package_resid'
    description = 'Resource with user_browse package'
    resource_type = resource_types.USERFEAT_USER_BROWSE_PACKAGE
    group = userdata_base_task.PACKAGES_GROUP_NAME


class IpregResourceParameter(sdk_parameters.ResourceSelector):
    name = 'ipreg_resource_id'
    description = 'Resource with IPREG and co'
    required = False  # don't require it yet
    resource_type = resource_types.IPREG_SET
    group = userdata_base_task.DATA_FILES_GROUP_NAME


class ClickDaemonKeysResource(sdk_parameters.ResourceSelector):
    name = 'clickdaemon_keys_resource_id'
    description = 'Resource with CLICKDAEMON_KEYS'
    required = False
    resource_type = resource_types.CLICKDAEMON_KEYS
    group = userdata_base_task.DATA_FILES_GROUP_NAME


class UserBrowseUpdateFastDataRun(userdata_base_task.Task):
    """
        Runs user_browse/scripts/proc_period_fast.py on a stored sample of 'fast' user_sessions
    """

    type = 'USER_BROWSE_UPDATE_FAST_DATA'

    execution_space = 35000

    yt_testable = True

    input_parameters = util.smart_join_params(
        userdata_base_task.Task.input_parameters,
        UserBrowsePackage,
        IpregResourceParameter,
        ClickDaemonKeysResource,
    )

    @util.lazy_property
    def dates_context(self):
        return self.get_state_attrs()

    def patch_config(self):
        self.config_patcher.patch(
            os.path.join(self.ctx['berkanavt'], 'user_browse/scripts/user_browse/user_browse_config.py'),
            {
                "projectRoot": os.path.join(self.ctx['berkanavt'], 'user_browse'),
                "fastSystemUser": self.get_system_user(),
            }
        )

    def get_project_bin_dir(self):
        return os.path.join(self.ctx["berkanavt"], "user_browse/bin")

    def process_mr_data(self):
        paths = self.get_common_pythonpaths()
        for suffix in "", "/common":
            paths.append(os.path.join(self.ctx["berkanavt"], "user_browse/scripts" + suffix))

        ctx = self.ctx.copy()
        ctx.update(self.dates_context)

        self.patch_config()

        ctx['tables_prefix'] = self.get_tables_prefix()
        ctx["pythonpath"] = ":".join(paths)

        cmd_tmpl = (
            "cd {berkanavt}/user_browse/scripts/user_browse;"
            "PYTHONPATH={pythonpath} "
            "MR_OPT= "
            "MR_CLUSTER_INFO={mr_cluster_info} " +
            "python ./proc_period_fast.py {target_timestamp} --mr_table_prefix {tables_prefix}"
        )

        if all([k in ctx for k in ['start_time', 'end_time', 'time_step']]):
            time_list = range(int(ctx['start_time']), int(ctx['end_time']), int(ctx['time_step']))
        else:
            time_list = [ctx['fast_sessions_timestamp']]

        conn = self.rem_client.connector()
        pr = util.ProcessRunner()
        tags = [
            "cluster={}_spy_log_sessions_fast_{}",
            "cluster={}_publish_fast_spy_log_sessions_{}"
        ]
        for target_timestamp in time_list:
            for tag_tmpl in tags:
                tag = tag_tmpl.format(self.ctx["mr_cluster"], target_timestamp)
                conn.Tag(tag).Set()
            pr.add(
                self.specific_logs_prefix("proc_period_fast.py." + str(target_timestamp)),
                cmd_tmpl,
                target_timestamp=target_timestamp,
                **ctx
            )
        pr.run()
        self.rem_client.wait_all_packets()

    def updated_result_attrs(self, attrs):
        ctx = self.dates_context
        for n in ['fast_sessions_timestamp', 'start_time', 'end_time', 'time_step']:
            if n in ctx:
                attrs['browse_' + n] = ctx[n]
        return attrs


__Task__ = UserBrowseUpdateFastDataRun
