import os

from sandbox.sandboxsdk.process import run_process
import sandbox.sandboxsdk.parameters as sdk_parameters

from sandbox.projects import resource_types
from sandbox.projects.common.userdata import userdata_base_task, util
from sandbox.projects.logs import resources as us_resources


class UserSearchPackage(sdk_parameters.LastReleasedResource):
    name = 'user_search_package_resid'
    description = 'Resource with user_search package'
    resource_type = resource_types.USERFEAT_USER_SEARCH_PACKAGE
    group = userdata_base_task.PACKAGES_GROUP_NAME


class IpregResourceParameter(sdk_parameters.ResourceSelector):
    name = 'ipreg_resource_id'
    description = 'Resource with IPREG and co'
    required = False  # don't require it yet
    resource_type = resource_types.IPREG_SET
    group = userdata_base_task.DATA_FILES_GROUP_NAME


class BlockstatDictResourceParameter(sdk_parameters.ResourceSelector):
    name = 'blockstat_dict_resource_id'
    description = 'Resource with blockstat.dict'
    required = False
    resource_type = us_resources.SESSIONS_BLOCKSTAT
    group = userdata_base_task.DATA_FILES_GROUP_NAME


class UserSearchUpdateFastDataRun(userdata_base_task.Task):
    """
        Runs user_search/scripts/proc_period_fast.py on a stored sample of 'fast' user_sessions
    """

    type = 'USER_SEARCH_UPDATE_FAST_DATA'

    execution_space = 65000
    node_chunk_store_quota = 45 << 30
    forbid_chunk_storage_in_tmpfs = True
    yt_testable = True

    input_parameters = util.smart_join_params(
        userdata_base_task.Task.input_parameters,
        UserSearchPackage,
        IpregResourceParameter,
        BlockstatDictResourceParameter,
    )

    @util.lazy_property
    def dates_context(self):
        return self.get_state_attrs()

    def patch_config(self):
        self.config_patcher.patch(
            os.path.join(self.ctx['berkanavt'], 'user_search/scripts/user_search/user_search_config.py'),
            {
                "projectRoot": os.path.join(self.ctx['berkanavt'], 'user_search'),
                "fastSystemUser": self.get_system_user()
            }
        )

    def get_project_bin_dir(self):
        return os.path.join(self.ctx["berkanavt"], "user_search/bin")

    def process_mr_data(self):
        ctx = self.ctx.copy()
        ctx.update(self.dates_context)

        self.patch_config()

        paths = self.get_common_pythonpaths()
        for suffix in "", "/common":
            paths.append(os.path.join(self.ctx["berkanavt"], "user_search/scripts" + suffix))

        ctx["tables_prefix"] = self.get_tables_prefix()
        ctx["pythonpath"] = ":".join(paths)

        cmd_tmpl = (
            "cd {berkanavt}/user_search/scripts/user_search;"
            "PYTHONPATH={pythonpath} "
            "MR_NET_TABLE= "
            "MR_USER= "
            "MR_OPT= "
            "MR_CLUSTER_INFO={mr_cluster_info} " +
            "python ./proc_period_fast.py {target_timestamp} --mr_table_prefix {tables_prefix}"
        )

        if all([k in ctx for k in ['start_time', 'end_time', 'time_step']]):
            time_list = range(int(ctx['start_time']), int(ctx['end_time']), int(ctx['time_step']))
        else:
            time_list = [ctx['fast_sessions_timestamp']]

        conn = self.rem_client.connector()
        for target_timestamp in time_list:
            tag = "cluster={}_publish_user_sessions_fast_incomplete_{}".format(
                self.ctx['mr_cluster'], target_timestamp
            )
            conn.Tag(tag).Set()
            cmd = cmd_tmpl.format(target_timestamp=target_timestamp, **ctx)
            run_process(
                cmd, shell=True, check=True, wait=True,
                log_prefix=self.specific_logs_prefix('proc_period_fast.py.' + str(target_timestamp))
            )

        self.rem_client.wait_all_packets()

    def updated_result_attrs(self, attrs):
        ctx = self.dates_context
        for n in ['fast_sessions_timestamp', 'start_time', 'end_time', 'time_step']:
            if n in ctx:
                attrs['search_' + n] = ctx[n]
        return attrs


__Task__ = UserSearchUpdateFastDataRun
