from os.path import join as pj
import os
import logging

import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.projects import resource_types
from sandbox.projects.logs import resources as us_resources
from sandbox.projects.common.userdata import userdata_base_task, util


class UserdataPackage(sdk_parameters.LastReleasedResource):
    name = 'userdata_package_resid'
    description = 'Resource with userdata package'
    resource_type = resource_types.USERFEAT_USERDATA_PACKAGE
    group = userdata_base_task.PACKAGES_GROUP_NAME


class UserBrowsePackage(sdk_parameters.LastReleasedResource):
    name = 'user_browse_package_resid'
    description = 'Resource with user_browse package'
    resource_type = resource_types.USERFEAT_USER_BROWSE_PACKAGE
    group = userdata_base_task.PACKAGES_GROUP_NAME


class UserSearchPackage(sdk_parameters.LastReleasedResource):
    name = 'user_search_package_resid'
    description = 'Resource with user_search package'
    resource_type = resource_types.USERFEAT_USER_SEARCH_PACKAGE
    group = userdata_base_task.PACKAGES_GROUP_NAME


class Ipreg(sdk_parameters.ResourceSelector):
    name = 'ipreg_resource_id'
    description = 'Resource with IPREG and co'
    required = False  # don't require it yet
    resource_type = resource_types.IPREG_SET
    group = userdata_base_task.DATA_FILES_GROUP_NAME


class BlockstatDict(sdk_parameters.ResourceSelector):
    name = 'blockstat_dict_resource_id'
    description = 'Resource with blockstat.dict'
    required = False
    resource_type = us_resources.SESSIONS_BLOCKSTAT
    group = userdata_base_task.DATA_FILES_GROUP_NAME


class GeoA(sdk_parameters.ResourceSelector):
    name = 'geoa_resource_id'
    description = "Resource with geoa.c2p"
    required = False
    resource_type = resource_types.GEOA_C2P
    group = userdata_base_task.DATA_FILES_GROUP_NAME


class ClickDaemonKeys(sdk_parameters.LastReleasedResource):
    name = 'clickdaemon_keys_resource_id'
    description = 'Resource with CLICKDAEMON_KEYS'
    required = False
    resource_type = resource_types.CLICKDAEMON_KEYS
    group = userdata_base_task.DATA_FILES_GROUP_NAME


class UserdataPrepFastRun(userdata_base_task.Task):
    """
        Runs userdata/scripts/prep_fast.py on a joined search and browse fast data for a day
    """

    type = 'USERDATA_PREP_FAST_RUN'

    execution_space = 75000
    node_chunk_store_quota = 50 << 30
    forbid_chunk_storage_in_tmpfs = True
    yt_testable = True

    input_parameters = util.smart_join_params(
        userdata_base_task.Task.input_parameters,
        UserdataPackage,
        UserBrowsePackage,
        UserSearchPackage,
        Ipreg,
        BlockstatDict,
        GeoA,
        ClickDaemonKeys,
    )

    @util.lazy_property
    def dates_context(self):
        return self.get_state_attrs()

    def patch_config(self):
        self.config_patcher.patch(
            pj(self.ctx["berkanavt"], "userdata/scripts/userdata/userdata_config.py"),
            {
                "projectRoot": os.path.join(self.ctx["berkanavt"], "userdata"),
                "fastSystemUser": self.get_system_user(),
                "fastPeriodDays": 1  # XXX
            }
        )

    def get_project_bin_dir(self):
        return pj(self.ctx["berkanavt"], "userdata/bin")

    def process_mr_data(self):
        ctx = self.ctx.copy()
        ctx.update(self.dates_context)
        self.patch_config()

        ctx['tables_prefix'] = self.get_tables_prefix()
        ctx["mr_env"] = self.get_client_environ_str()

        conn = self.rem_client.connector()

        for sys, tag_sys in ("search", "user_search"), ("browse", "browse"):
            time_list = range(
                int(ctx[sys + '_start_time']),
                int(ctx[sys + '_end_time']),
                int(ctx[sys + '_time_step'])
            )

            for target_timestamp in time_list:
                tag = "cluster={}_{}_factors_fast_{}".format(
                    self.ctx["mr_cluster"],
                    tag_sys,
                    target_timestamp
                )
                logging.debug("Setting tag " + tag)
                conn.Tag(tag).Set()

        paths = self.get_common_pythonpaths()
        for suffix in "", "/common":
            paths.append(os.path.join(self.ctx["berkanavt"], "userdata/scripts" + suffix))
        ctx["pythonpath"] = ":".join(paths)

        cmd_tmpl = (
            "cd {berkanavt}/userdata/scripts/userdata;"
            "PYTHONPATH={pythonpath} "
            "MR_NET_TABLE= "
            "MR_USER= "
            "MR_OPT= "
            "MR_CLUSTER_INFO={mr_cluster_info} "
            "{mr_env} "
            "python ./prep_fast.py "
            "--base_ts {search_end_time} "
            "--mr_table_prefix {tables_prefix} "
            "--user_browse_start_ts {browse_start_time} "
            "--user_browse_end_ts {browse_end_time} "
            "--user_search_start_ts {search_start_time} "
            "--user_search_end_ts {search_end_time} "
        )
        if self.test_requirements(
            "USERFEAT_USERDATA_PACKAGE",
            trunk_revision=3634336,
            branch_num=8,
        ):
            cmd_tmpl += "--dont_publish_to_reactor "

        util.run_shell_process("prep_fast.py", cmd_tmpl, **ctx)

        self.rem_client.wait_all_packets()

    def updated_result_attrs(self, attrs):
        ctx = self.dates_context
        for src in ['search', 'browse']:
            for n in ['fast_sessions_timestamp', 'start_time', 'end_time', 'time_step']:
                fn = src + '_' + n
                if fn in ctx:
                    attrs[fn] = ctx[fn]
        return attrs


__Task__ = UserdataPrepFastRun
