import json
import logging
import os

from sandbox import sdk2
from sandbox.common.errors import TaskFailure


def find_resource(search_conf):
    logging.info('lookup resource str: %s' % search_conf)
    try:
        params = json.loads(search_conf)
    except Exception as exc:
        raise ValueError('Error while loading resource search config: {}, {}'
                         .format(search_conf, exc))
    logging.info('find_resource with params: {}'.format(params))
    res = sdk2.Resource.find(**params).order(-sdk2.Resource.id).first()
    logging.info('resource found: id: {}, size: {}, desc: "{}"'.format(res.id,
                                                                       res.size,
                                                                       res.description))
    return res


class ExecScriptBase(sdk2.Task):
    """A base task, similar in purpose to RUN_SCRIPT, with sdk2.
    but allow to delay failure until delved class collect artifacts produced
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 8
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        out_resource_list = []

    class Parameters(sdk2.Task.Parameters):
        command = sdk2.parameters.String('Command to execute: "docker ps -a > docker.log; lscpu > lscpu.txt;"', required=True, multiline=True)
        vault_env = sdk2.parameters.Dict('Vault items to put in the environment', required=False)

    # Derived class may delay this exception
    def on_exception(self, exception):
        raise exception

    def get_workspace(self):
        return self.path('script')

    def dump_file_tail(self, fname, header, skip_empty=True, log_op=None, num_lines=10):
        if skip_empty:
            if not os.path.exists(fname) or not os.path.getsize(fname):
                return
        if not log_op:
            log_op = self.set_info

        lines = [header + '\n']
        with open(fname) as fp:
            lines += fp.readlines()[-num_lines:]
        log_op("".join(lines))

    def on_execute(self):
        scriptdir = str(self.get_workspace()).strip()
        used_resources = []
        resources = dict()
        if used_resources:
            ur_list = []
            for r in used_resources:
                ur_list.append(r.id)
            self.Parameters.used_resources = ur_list

        if not os.path.exists(scriptdir):
            os.makedirs(scriptdir)

        command_env = os.environ.copy()
        for vault_item, env_name in self.Parameters.vault_env.items():
            command_env[env_name] = sdk2.Vault.data(vault_item)

        if resources:
            command = self.Parameters.command.format(**resources)
        else:
            command = self.Parameters.command

        logging.info('Run command  "%s"', command)
        with sdk2.helpers.ProcessLog(self, logger="script") as pl:
            proc = sdk2.helpers.subprocess.Popen(command,
                                                 env=command_env,
                                                 cwd=scriptdir,
                                                 shell=True,
                                                 stdout=pl.stdout,
                                                 stderr=pl.stderr,
                                                 close_fds=True)
            url = "%s/%s" % (self.log_resource.http_proxy,
                             pl.stdout.path.relative_to(self.path(self.log_resource.path)))
            err_url = "%s/%s" % (self.log_resource.http_proxy,
                                 pl.stderr.path.relative_to(self.path(self.log_resource.path)))
            tail_sfx = '?tail=1&force_text_mode=1'
            self.set_info('Process started. <a href="{0}" target="_blank">output</a>, <a href="{1}" target="_blank">stderr</a>'.format(url, err_url),
                          do_escape=False)
            self.set_info('Process <a href="{0}{1}">tail log </a>'.format(url, tail_sfx), do_escape=False)
            logging.info('wait task')
            proc.wait()
            logging.error('Script err_code:%d', proc.returncode)
            if proc.returncode:
                pl.stdout.flush()
                pl.stderr.flush()
                self.dump_file_tail(str(pl.stdout.path), "--------------------------- Captured stdout call ---------------------------")
                self.dump_file_tail(str(pl.stderr.path), "--------------------------- Captured stderr call ---------------------------")
                self.on_exception(TaskFailure('Script fail with %d' % proc.returncode))
            else:
                self.set_info('Script succeed')
