from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import dolbilka
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common import utils
from sandbox.projects import PatchPlan as patch_task
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.search.basesearch import task as search_task


_BASESEARCH_PARAMS = search_components.create_noweb_basesearch_params(archive_model_required=False)
_VARIANT_TASKS = 'variants'


class PlanParameter(parameters.ResourceSelector):
    name = 'dolbilo_plan_resource_id'
    description = 'Plan for basesearch'
    resource_type = (resource_types.BASESEARCH_PLAN,)
    group = dolbilka.DOLBILKA_GROUP
    required = True


class CgiParamsParameter(parameters.SandboxStringParameter):
    name = 'cgi_params'
    description = 'Additional cgi parameters (each line for separate experiment)'
    default_value = ''
    multiline = True
    group = dolbilka.DOLBILKA_GROUP


class VideoAnalyzeBasesearchCgiParams(search_performance.NewShootingTask, search_task.BasesearchComponentTask):
    """
        Analyze basesearch performance with different cgi params
    """

    type = 'VIDEO_ANALYZE_BASESEARCH_CGI_PARAMS'

    client_tags = search_performance.NewShootingTask.client_tags & search_task.BasesearchComponentTask.client_tags

    basesearch_input_parameters = \
        search_task.BasesearchComponentTask.basesearch_input_parameters + \
        _BASESEARCH_PARAMS.params

    shoot_input_parameters = \
        (PlanParameter, CgiParamsParameter,) + \
        search_performance.NewShootingTask.shoot_input_parameters

    input_parameters = basesearch_input_parameters + shoot_input_parameters

    def on_execute(self):
        search_task.BasesearchComponentTask.on_execute(self)
        media_settings.VideoSettings.ensure_search_database(self.ctx, PlanParameter, _BASESEARCH_PARAMS.Database)

        if _VARIANT_TASKS not in self.ctx:
            plan_resource_id = self.ctx[PlanParameter.name]
            variants = [("#0: original plan", plan_resource_id)]
            for n, cgi_params in enumerate(self.ctx[CgiParamsParameter.name].split('\n'), 1):
                if cgi_params:
                    variants.append(("#{}: {}".format(n, cgi_params), self._patch_plan(plan_resource_id, cgi_params)))
            self.ctx[_VARIANT_TASKS] = variants

        self._sync_subtasks()

        self._init_virtualenv()
        for variant, plan_resource_id in self.ctx[_VARIANT_TASKS]:
            self._dolbilo_shoot(self._basesearch(_BASESEARCH_PARAMS), plan_resource_id, variant)

    def _sync_subtasks(self):
        subtasks = self.list_subtasks()
        has_incomplete_tasks = not all([channel.sandbox.get_task(x).is_done() for x in subtasks])
        if has_incomplete_tasks:
            self.wait_all_tasks_completed(subtasks)

        utils.check_if_tasks_are_ok(subtasks)

    def _patch_plan(self, plan_resource_id, cgi_params):
        sub_ctx = {
            patch_task.PlanParameter.name: plan_resource_id,
            patch_task.AddCgiParameter.name: cgi_params,
        }
        sub_task = self.create_subtask(
            task_type=patch_task.PatchPlan.type,
            input_parameters=sub_ctx,
            description="{}, {}".format(self.descr, cgi_params)
        )
        new_plan_resources = apihelpers.list_task_resources(
            sub_task.id, resource_type=resource_types.BASESEARCH_PLAN, limit=1)
        if not new_plan_resources:
            raise errors.SandboxTaskFailureError("Failed to find plan inside {} task".format(sub_task.id))

        return new_plan_resources[0].id


__Task__ = VideoAnalyzeBasesearchCgiParams
