import json
import urllib

from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka
from sandbox.projects.common import profiling as search_profiling
from sandbox.projects.common.differ import coloring
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.search.basesearch import task as search_task


# Used for profile file names
_FIRST_RUN = 0
_SECOND_RUN = 1

_FIRST_VARIANT = "#{}".format(_FIRST_RUN+1)
_SECOND_VARIANT = "#{}".format(_SECOND_RUN+1)
_DIFF_VARIANT = "diff"

_MIN_RAM = 96

BASESEARCH1_PARAMS = search_components.create_noweb_basesearch_params(n=1, archive_model_required=False)
BASESEARCH2_PARAMS = search_components.create_noweb_basesearch_params(n=2, archive_model_required=False)


class Plan1Parameter(parameters.ResourceSelector):
    name = 'dolbilo_plan1_resource_id'
    description = 'Plan for basesearch #1'
    resource_type = resource_types.BASESEARCH_PLAN
    group = dolbilka.DOLBILKA_GROUP
    required = True


class Plan2Parameter(parameters.ResourceSelector):
    name = 'dolbilo_plan2_resource_id'
    description = 'Plan for basesearch #2'
    resource_type = resource_types.BASESEARCH_PLAN
    group = dolbilka.DOLBILKA_GROUP
    required = True


class VideoAnalyzeBasesearchPerformance(
        search_performance.NewShootingTask,
        search_profiling.ProfilingTask,
        search_task.BasesearchComponentTask):
    """
        Analyze basesearch performance
    """

    type = 'VIDEO_ANALYZE_BASESEARCH_PERFORMANCE'

    basesearch_input_parameters = \
        search_task.BasesearchComponentTask.basesearch_input_parameters + \
        BASESEARCH1_PARAMS.params + \
        BASESEARCH2_PARAMS.params

    shoot_input_parameters = \
        (Plan1Parameter, Plan2Parameter,) + \
        search_performance.NewShootingTask.shoot_input_parameters

    input_parameters = \
        basesearch_input_parameters + \
        shoot_input_parameters + \
        search_profiling.ProfilingTask.input_parameters

    execution_space = 70 * 1024
    required_ram = _MIN_RAM * 1024

    client_tags = search_performance.NewShootingTask.client_tags

    @property
    def footer(self):
        footers = []

        performance_footer = search_performance.NewShootingTask.footer.fget(self)
        footers.append({
            'helperName': '',
            'content': performance_footer
        })

        profiling_footer = search_profiling.ProfilingTask.profiling_footer(self)
        if profiling_footer:
            footers.append({
                'helperName': '',
                'content': profiling_footer
            })

        return footers

    def on_enqueue(self):
        search_task.BasesearchComponentTask.on_enqueue(self)
        media_settings.VideoSettings.ensure_search_database(
            self.ctx,
            Plan1Parameter,
            BASESEARCH1_PARAMS.Database
        )
        media_settings.VideoSettings.ensure_search_database(
            self.ctx,
            Plan2Parameter,
            BASESEARCH2_PARAMS.Database
        )

    def on_execute(self):
        search_task.BasesearchComponentTask.on_execute(self)

        self._init_virtualenv()
        self._dolbilo_shoot(BASESEARCH1_PARAMS, Plan1Parameter, _FIRST_VARIANT, _FIRST_RUN)
        self._dolbilo_shoot(BASESEARCH2_PARAMS, Plan2Parameter, _SECOND_VARIANT, _SECOND_RUN)

        stats = self.ctx[self.new_stats_key]
        for key in stats[_FIRST_VARIANT]:
            if self._in_stats(key):
                stats.setdefault(_DIFF_VARIANT, {})[key] = _delta(
                    stats[_FIRST_VARIANT][key],
                    stats[_SECOND_VARIANT][key]
                )
        self._profiling_diff(
            self.__get_perf_path(_FIRST_VARIANT),
            self.__get_perf_path(_SECOND_VARIANT),
            self.__get_perf_path(_DIFF_VARIANT),
            description="diff"
        )
        self.ctx['{}_json'.format(self.new_stats_key)] = json.dumps(self.ctx[self.new_stats_key])

    def _dolbilo_shoot(self, basesearch_params, plan_parameter, variant, run_count):
        media_settings.VideoSettings.ensure_search_database(
            self.ctx,
            plan_parameter,
            basesearch_params.Database
        )

        basesearch = self._basesearch(basesearch_params, run_count=run_count)
        data_dir = self.__get_perf_path(variant)

        self._profiling_init(basesearch, data_dir)
        search_performance.NewShootingTask._dolbilo_shoot(
            self,
            basesearch,
            self.ctx[plan_parameter.name],
            variant
        )
        self._profiling_report(basesearch, data_dir, description=variant)

    def _format_stats(self, fmt, stats, variant, key):
        if key == "shooting.rps_0.5" and variant == _DIFF_VARIANT:
            return coloring.color_diff(stats[variant][key], max_diff=-1)
        elif key.startswith("shooting.latency_") and variant == _DIFF_VARIANT:
            return coloring.color_diff(stats[variant][key], max_diff=1)
        else:
            txt = search_performance.NewShootingTask._format_stats(self, fmt, stats, variant, key)
            return txt + ("%" if variant == _DIFF_VARIANT else "")

    def __get_perf_path(self, variant):
        return self.abs_path("perf.{}.data".format(urllib.quote(variant, safe="")))


def _delta(a, b):
    return (float(b - a) / a) * 100 if a else 0


__Task__ = VideoAnalyzeBasesearchPerformance
