# -*- coding: utf-8 -*-

import shutil
import sys

from sandbox.common import rest
from sandbox.projects import resource_types
from sandbox.projects import BuildDynamicModels as bdm
from sandbox.projects import VideoBuildDynamicModels
from sandbox.projects import VideoTestDynamicModelsArchive as vtdma

from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel


def _copy_archive(source_archive_id, destination_archive_id):
    """
        Копирует данные из одного ресурса архива в другой
    """

    source_resource_path = channel.task.sync_resource(source_archive_id)
    destination_resource = channel.sandbox.get_resource(destination_archive_id)
    shutil.copy(source_resource_path, destination_resource.path)
    channel.task.mark_resource_ready(destination_resource)


class VideoBuildAndTestDynamicModels(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Таск сначала собирает архив с моделями
        из указанного источника, а затем
        тестирует полученный архив.

        Для сборки применяется таск VIDEO_BUILD_DYNAMIC_MODELS
        Для тестирования применяется таск VIDEO_TEST_DYNAMIC_MODULES_ARCHIVE
    """

    type = 'VIDEO_BUILD_AND_TEST_DYNAMIC_MODELS'

    archive_type = resource_types.VIDEO_DYNAMIC_MODELS_ARCHIVE

    build_task_type = 'VIDEO_BUILD_DYNAMIC_MODELS'

    test_task_type = 'VIDEO_TEST_DYNAMIC_MODELS_ARCHIVE'

    input_parameters = VideoBuildDynamicModels.VideoBuildDynamicModels.input_parameters + (
        vtdma.UseCustomVideosearchParameter,
        vtdma.VideosearchParameter,
    )

    def arcadia_info(self):
        """
            Возвращает информацию о SVN репозитории с моделями
        """
        return bdm.dynamic_models_archive_arcadia_info(self.ctx)

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self
        resource = self.create_resource(
            'Dynamic models archive for videosearch',
            'videosearch_models.archive',
            self.archive_type,
            arch='any'
        )
        self.ctx['out_resource_id'] = str(resource.id)

    def on_execute(self):
        if 'build_task_id' not in self.ctx:
            self.ctx['build_task_id'] = str(self._build_archive())

        build_task_id = int(self.ctx['build_task_id'])
        build_task = channel.sandbox.get_task(build_task_id)
        if not build_task.is_done():
            self.wait_task_completed(build_task_id)

        if not self.ctx[vtdma.UseCustomVideosearchParameter.name]:
            self.ctx['videosearch_resource_id'] = None

        if 'test_task_id' not in self.ctx:
            print >> sys.stderr, build_task.ctx
            utils.check_if_tasks_are_ok([build_task_id])
            self.ctx['diff'] = build_task.ctx.get('diff')
            self.ctx['archive_resource_id'] = str(build_task.ctx['out_resource_id_string'])
            self.ctx['test_task_id'] = str(self._test_archive(self.ctx['videosearch_resource_id'], int(self.ctx['archive_resource_id'])))

        test_task_id = int(self.ctx['test_task_id'])
        test_task = channel.sandbox.get_task(test_task_id)
        if not test_task.is_done():
            self.wait_task_completed(test_task_id)

        utils.check_if_tasks_are_ok([test_task_id])
        _copy_archive(int(self.ctx['archive_resource_id']), int(self.ctx['out_resource_id']))

    def _build_archive(self):
        """
            Запустить подзадачу для сборки архива
        """

        sub_ctx = {
            bdm.ModelsForProduction.name:       self.ctx[bdm.ModelsForProduction.name],
            bdm.ModelsUrlForProduction.name:    self.ctx[bdm.ModelsUrlForProduction.name],
            bdm.ModelsForExperiment.name:       self.ctx[bdm.ModelsForExperiment.name],
            bdm.ModelsUrlForExperiment.name:    self.ctx[bdm.ModelsUrlForExperiment.name],
            bdm.CheckProductionModelsList.name: self.ctx[bdm.CheckProductionModelsList.name],
            bdm.StrictFormulaIdChecks.name:     self.ctx[bdm.StrictFormulaIdChecks.name],
            bdm.AnnotateSlices.name:            False,
        }
        sub_task = self.create_subtask(
            task_type=self.build_task_type,
            input_parameters=sub_ctx,
            description=self.descr,
        )
        return sub_task.id

    def _test_archive(self, videosearch_id, archive_id):
        """
            Запустить подзадачу для тестирования архива
        """

        sub_ctx = {
            vtdma.ModelsArchiveParameter.name: archive_id,
            vtdma.UseCustomVideosearchParameter.name: videosearch_id is not None,
            vtdma.VideosearchParameter.name: videosearch_id
        }
        sub_task = self.create_subtask(
            task_type=self.test_task_type,
            input_parameters=sub_ctx,
            description=self.descr,
        )
        return sub_task.id

    @property
    def footer(self):
        client = rest.Client()
        children = self.list_subtasks()
        if len(children) < 2:
            return None
        return client.task[children[1]].custom.footer.read()


__Task__ = VideoBuildAndTestDynamicModels
