import xml.etree.ElementTree as xmlET
import xml.dom.minidom as dom
import re
import time
from collections import defaultdict


class PlayerCollector():
    HOSTS_PER_CONNECTOR = 256

    class Player():
        def __init__(self, player_dict):
            self.p_source = player_dict["PlayerSource"]
            self.p_type = player_dict["PlayerType"]
            self.timestamp = player_dict["CreatedTimestamp"]
            self.result = player_dict["Result"]
            self.no_popups = player_dict["NoPopups"] if "NoPopups" in player_dict else None
            self.host = player_dict["Host"]
            self.platform = player_dict["Platform"] if "Platform" in player_dict else "All"
            self.params = (("sandbox", not self.no_popups if self.no_popups is not None else False),)

    def get_key(self, player):
        return (player.p_source, player.p_type)

    def get_player_groups(self):
        return self.connectors_types

    def __init__(self, ytc):
        self.ytc = ytc
        self.query = "* FROM [{0}] WHERE Result = True"
        self.connectors_types = defaultdict(lambda: defaultdict(lambda: defaultdict(dict)))

    def add_players(self, src_table_path, prefix):
        data = self.ytc.select_rows(self.query.format(src_table_path))
        self.add_connectors(data, prefix)

    def create_united_player(self, platforms):
        united_player_data = {
            "Result": False,
            "NoPopups": True
        }

        for platform, player in platforms.iteritems():
            united_player_data["Result"] = united_player_data["Result"] or player.result
            united_player_data["NoPopups"] = united_player_data["NoPopups"] and player.no_popups
            united_player_data["PlayerSource"] = player.p_source
            united_player_data["PlayerType"] = player.p_type
            united_player_data["Host"] = player.host
            united_player_data["CreatedTimestamp"] = player.timestamp

        return self.Player(united_player_data)

    def get_united_platform_params(self, connectors):
        for key, hosts in connectors.iteritems():
            for host in hosts:
                hosts[host] = self.create_united_player(hosts[host])

    def add_connectors(self, data, prefix):
        for line in data:
            player = self.Player(line)
            platform = player.platform
            self.connectors_types[prefix][self.get_key(player)][player.host][platform] = player

        self.get_united_platform_params(self.connectors_types[prefix])

    def create_connectors_xml(self):
        root = xmlET.Element('connectors')

        for prefix, connectors in self.connectors_types.iteritems():
            for key, hosts in connectors.iteritems():
                params_dict = defaultdict(dict)

                for host, player in hosts.iteritems():
                    params_dict[player.params][host] = player

                for params, hosts in params_dict.iteritems():
                    host_vec = hosts.items()
                    for i in range(len(host_vec) / self.HOSTS_PER_CONNECTOR + 1):
                        host_portion = host_vec[i * self.HOSTS_PER_CONNECTOR : (i + 1) * self.HOSTS_PER_CONNECTOR]
                        if host_portion:
                            self.create_connector(root, dict(host_portion), key, prefix, params)

        return dom.parseString(xmlET.tostring(root, encoding='utf8', method='xml')).toprettyxml(indent="    ")

    def create_connector(self, root, hosts, key, prefix, params):
        player_source = key[0]
        player_type = key[1]

        connector = xmlET.SubElement(root, "connector")
        connector.set("player_id", prefix + player_source + "_" + player_type + "__")
        connector.set("priority", "50")

        url_node = xmlET.SubElement(connector, "match")
        url_node.set("string", r"$(url)")
        url_node.set("regex", self.create_regexp(hosts))

        source_node = xmlET.SubElement(connector, "match")
        source_node.set("string", r"$(source)")
        source_node.set("regex", r"^" + player_source + r"$")

        type_node = xmlET.SubElement(connector, "match")
        type_node.set("string", r"$(contents.type)")
        type_node.set("regex", r"^" + player_type + r"$")

        contents_node = xmlET.SubElement(connector, "match")
        contents_node.set("string", r"$(contents.url)")
        contents_node.set("regex", r"^\s*(?:(?:https?:)?\/\/)?([\S]+)\s*$")

        group_node = xmlET.SubElement(contents_node, "group")
        group_node.set("number", "1")
        group_node.set("to_param", "url")

        created_node = xmlET.SubElement(connector, "created")
        created_node.set("at", "2008-01-01")

        for host in hosts:
            player = hosts[host]
            release_node = xmlET.SubElement(created_node, "added")
            release_node.set("host", player.host)
            release_node.set("at", str(self.get_player_release_date(player, key)))

        for param in params:
            param_node = xmlET.SubElement(connector, "param")
            param_node.set("name", param[0])
            param_node.set("value", str(param[1]))

    def create_regexp(self, hosts):
        hosts = [re.escape(host) for host in hosts]
        return r"^(?:" + "|".join(hosts) + r")\/"

    def get_player_release_date(self, player, key):
        tm = time.gmtime(player.timestamp)

        return "-".join([str(tm.tm_year), "%02d" % tm.tm_mon, "%02d" % tm.tm_mday])
