# -*- coding: utf-8 -*-

import logging
import xmlrpclib

from sandbox.sandboxsdk.channel import channel

from sandbox.projects.MediaLib.iss_shardtracker import iss_shardtracker_api


CC_USERS = [
    "videosearch-releases",
    "nextus",
    "vbiriukov",
    "matveieff",
]

PRIEMKA_TASK = 'PRIEMKA_VIDEO_BASESEARCH_DATABASE'
SHARDTRACKER = ('shardtracker.search.yandex.net', 9100)

# Shard size limit (in Gb)
LOW_SIZE_LIMIT = 19
HIGH_SIZE_LIMIT = 40.8


class VideoAccept:
    def __init__(self, parent_task, shardmap):
        self.shardmap = shardmap
        self.parent = parent_task
        self.shardmap_timestamp = self.shardmap.shardmap_timestamp
        self.shardmap_name = self.shardmap.shardmap_filename
        self.message = "Database: {0}\n".format(self.shardmap_name)
        self.message += "Timestamp: {0}\n\n".format(self.shardmap_timestamp)

    def _check_shardmap_sizes(self):
        """
        Проверка размера шарда
        """
        # Shard size limit (in Gb)
        lower_size_limit = LOW_SIZE_LIMIT
        upper_size_limit = HIGH_SIZE_LIMIT

        shard_sizes = []

        with iss_shardtracker_api() as thrift_shardtracker_client:
            for shard_name in self.shardmap.shards:
                shard_info = thrift_shardtracker_client.getShardInfo(shard_name)
                shard_size = float(shard_info.shardInfo.full.size)/2**30
                shard_sizes.append(shard_size)

                if shard_size < lower_size_limit or shard_size > upper_size_limit:
                    raise Exception("Shard size is incorrect: {0:.2f}GB, shard {1}\n".format(shard_size, shard_name))

        shards_count = len(self.shardmap.shards)
        info = """
Shard sizes are OK.
Shards count: {shards_count}
Shard sizes(min/avg/max): {min:.2f}/{avg:.2f}/{max:.2f}GB
Total index size: {total_size:.2f}GB
""".format(
            min=min(shard_sizes),
            avg=sum(shard_sizes) / shards_count,
            max=max(shard_sizes),
            shards_count=shards_count,
            total_size=sum(shard_sizes)
        )

        self.parent.set_info(info, do_escape=False)
        logging.debug(info)
        self.message += info

    def __check_performance_single_task(self, task, proxy):
        tasks_run = 0

        local_info = "Performance check task: <a href='https://sandbox.yandex-team.ru/task/{0}/view'>Task {0}</a><br/>".format(task["id"])
        for task_name, task_id in task["ctx"]["analyze_tasks"].iteritems():
            subtask = proxy.get_task(task_id)
            median = subtask["ctx"]["new_stats"]["diff"]["shooting.rps_0.5"]
            tasks_run += 1

            local_info += "\tMedian diff {2:.2f}% \t{1:<18}\t<a href='https://sandbox.yandex-team.ru/task/{0}/view'>{0}</a><br/>".format(task_id, task_name.split(",")[-2:], median)
            if median < -10:
                raise Exception("Median diff exceeds -10%")
        self.message += local_info
        self.parent.set_info(local_info, do_escape=False)

        return tasks_run

    def _check_performance(self):
        r = xmlrpclib.ServerProxy('https://sandbox.yandex-team.ru/sandbox/xmlrpc')
        t = r.list_tasks({"status": "SUCCESS", "task_type": PRIEMKA_TASK, "show_childs": True})

        found_task = 0
        for task in t:
            wrong_task = 0

            if "new_basesearch_database_shards" not in task["ctx"]:
                continue

            for shard in task["ctx"]["new_basesearch_database_shards"].split(","):
                if shard.find(self.shardmap_timestamp) == -1:
                    wrong_task = 1
                    break

            if wrong_task:
                continue

            found_task += self.__check_performance_single_task(task, r)

        if not found_task:
            raise Exception("No valid {0} task found for {1}".format(PRIEMKA_TASK, self.shardmap_timestamp))

    def accept(self):
        logging.debug("Check db performance")
        self._check_performance()
        logging.debug("Performance is ok")

        logging.debug("Checking shard sizes")
        self._check_shardmap_sizes()
        logging.debug("Shard sizes are ok")

        channel.sandbox.send_email(
            'vbiriukov@yandex-team.ru',
            CC_USERS,
            'Видеобаза {0} принята по производительности'.format(self.shardmap_name),
            self.message,
            'text/html',
            'utf-8'
        )

        logging.debug("Acceptance done")
