# -*- coding: utf-8 -*-

import time
import logging

from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters

from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.common.middlesearch import video_single_host

from sandbox.projects.images.metasearch import components as randomizers


class VideoSearchRandomizer(randomizers.ImgSearchRandomizer):
    """
        Используем ImgSearchRandomizer вместо написания собственного класса
    """
    pass


_RESPONSE_SAVER_PARAMS = response_saver.create_response_saver_params(
    queries_resource=resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS
)


class InitialSeedParameter(parameters.SandboxIntegerParameter):
    name = 'initial_seed'
    description = 'Initial random seed (use 0 for current time)'
    default_value = 0


class GtasParameter(parameters.SandboxStringParameter):
    name = 'randomize_gtas'
    description = 'Attributes to randomize (use "" to randomize all attributes)'


class EnableRandomizeParameter(parameters.SandboxBoolParameter):
    name = 'enable_randomize'
    description = 'Randomize output from basesearch'
    default_value = False
    sub_fields = {
        'true': [
            InitialSeedParameter.name,
            GtasParameter.name
        ]
    }


class VideoGetMiddlesearchResponses(video_single_host.VideoMiddlesearchSingleHostTask):
    """
        Запускает один средний и два базовых поиска
        Ответы среднего сохраняются в ресурсе типа BASESEARCH_HR_RESPONSES

        Основное отличие от GET_MIDDLESEARCH_RESPONSES - возможность автоматического
        определения ресурсов индексов для базовых поисков
    """

    type = 'VIDEO_GET_MIDDLESEARCH_RESPONSES'

    input_parameters = \
        video_single_host.VideoMiddlesearchSingleHostTask.input_parameters \
        + _RESPONSE_SAVER_PARAMS.params \
        + (EnableRandomizeParameter, GtasParameter, InitialSeedParameter) \
        + threadPool.PARAMS

    def on_execute(self):
        initial_seed = self.ctx.get(InitialSeedParameter.name, InitialSeedParameter.default_value)
        if initial_seed == InitialSeedParameter.default_value:
            self.ctx[InitialSeedParameter.name] = int(time.time())
        video_single_host.VideoMiddlesearchSingleHostTask.on_execute(self)

    def on_enqueue(self):
        video_single_host.VideoMiddlesearchSingleHostTask.on_enqueue(self)
        response_saver.create_resources(self)

    def _get_queries_parameter(self):
        return _RESPONSE_SAVER_PARAMS.QueriesParameter

    def _get_basesearch_ports_for_middlesearch(self):
        base1_port, base2_port = video_single_host.VideoMiddlesearchSingleHostTask._get_basesearch_ports_for_middlesearch(self)

        if not self.ctx.get(EnableRandomizeParameter.name, EnableRandomizeParameter.default_value):
            return base1_port, base2_port

        return base1_port + 10, base2_port + 10

    def _use_middlesearch_component(self, middlesearch):
        patched_queries_resource = response_saver.get_patched_queries_resource(self)

        def wrapper(res, prepare_session=None):
            response_saver.save_responses(
                self.ctx,
                search_component=middlesearch,
                responses_resource=res,
                patched_queries_resource=patched_queries_resource,
                prepare_session_callback=prepare_session,
            )

        if self.ctx.get(EnableRandomizeParameter.name, EnableRandomizeParameter.default_value):
            gtas = self.ctx.get(GtasParameter.name, GtasParameter.default_value)
            gtas = gtas.split(",") if gtas else []

            initial_seed = self.ctx[InitialSeedParameter.name]

            proxy1_port, proxy2_port = self._get_basesearch_ports_for_middlesearch()
            base1_port, base2_port = video_single_host.VideoMiddlesearchSingleHostTask._get_basesearch_ports_for_middlesearch(self)
            with VideoSearchRandomizer(proxy1_port, base1_port, gtas=gtas, initial_seed=initial_seed):
                with VideoSearchRandomizer(proxy2_port, base2_port, gtas=gtas, initial_seed=initial_seed):
                    #  if need to test responses correctness with cache disabled:
                    if self.ctx.get(_RESPONSE_SAVER_PARAMS.TestWithCacheDisabled.name):
                        logging.info("Getting responses with disabled cache")
                        wrapper(
                            channel.sandbox.get_resource(self.ctx['without_cache_responses_id']),
                            middlesearch.disable_cache
                        )
                    logging.info("Getting responses with empty enabled cache")
                    wrapper(channel.sandbox.get_resource(self.ctx['out_resource_id']), middlesearch.clear_cache)
        else:
            #  if need to test responses correctness with cache disabled:
            if self.ctx.get(_RESPONSE_SAVER_PARAMS.TestWithCacheDisabled.name):
                logging.info("Getting responses with disabled cache")
                wrapper(
                    channel.sandbox.get_resource(self.ctx['without_cache_responses_id']),
                    middlesearch.disable_cache
                )
            logging.info("Getting responses with empty enabled cache")
            wrapper(channel.sandbox.get_resource(self.ctx['out_resource_id']), middlesearch.clear_cache)


__Task__ = VideoGetMiddlesearchResponses
