# -*- coding: utf-8 -*-_read_resource

import json
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.projects.common import apihelpers
from sandbox.projects import resource_types


class VideoIndexTestingStateParameter(parameters.SandboxStringParameter):
    name = "testing_state"
    description = "testing state"
    required = True


class VideoIndexStableStateParameter(parameters.SandboxStringParameter):
    name = "stable_state"
    description = "stable state"
    default_value = "stable"


class VideoIndexValidation(task.SandboxTask):
    """
        Validates video index
    """

    type = 'VIDEO_INDEX_VALIDATION'

    input_parameters = (
        VideoIndexTestingStateParameter,
        VideoIndexStableStateParameter,
    )

    def on_execute(self):

        if 'stable_resource_id' not in self.ctx and 'stable_subtask_id' not in self.ctx:
            stable_state = self.ctx[VideoIndexStableStateParameter.name]

            stable_resource = None
            if 'stable_subtask_id' not in self.ctx:
                stable_resource = self._get_resource(stable_state)

            if not stable_resource:
                self.set_info('stable resource not found. Creating subtask...')
                stable_subtask = self._make_info_subtask(stable_state)
                self.ctx['stable_subtask_id'] = stable_subtask.id

                self.wait_tasks(
                    self.ctx['stable_subtask_id'],
                    (self.Status.SUCCESS, self.Status.FAILURE, self.Status.EXCEPTION),
                    wait_all=True,
                    state='Waiting for stable info to complete',
                )
            else:
                self.ctx['stable_info_id'] = stable_resource.id

        if 'testing_resource_id' not in self.ctx and 'testing_subtask_id' not in self.ctx:
            testing_state = self.ctx[VideoIndexTestingStateParameter.name]

            testing_resource = None
            if 'testing_subtask_id' not in self.ctx:
                testing_resource = self._get_resource(testing_state)

            if not testing_resource:
                self.set_info('testing resource not found. Creating subtask...')
                testing_subtask = self._make_info_subtask(testing_state)
                self.ctx['testing_subtask_id'] = testing_subtask.id

                self.wait_tasks(
                    self.ctx['testing_subtask_id'],
                    (self.Status.SUCCESS, self.Status.FAILURE, self.Status.EXCEPTION),
                    wait_all=True,
                    state='Waiting for testing info to complete',
                )
            else:
                self.ctx['testing_info_id'] = testing_resource.id

        if self.ctx['stable_subtask_id']:
            self.ctx['stable_info_id'] = apihelpers.get_task_resource_id(
                self.ctx['stable_subtask_id'],
                resource_types.VIDEO_INDEX_INFO,
            )

        if self.ctx['testing_subtask_id']:
            self.ctx['testing_info_id'] = apihelpers.get_task_resource_id(
                self.ctx['testing_subtask_id'],
                resource_types.VIDEO_INDEX_INFO,
            )

        stable_info = None
        with open(self.sync_resource(self.ctx['stable_info_id'])) as f:
            stable_info = json.loads(f.read(), encoding='utf-8')

        testing_info = None
        with open(self.sync_resource(self.ctx['testing_info_id'])) as f:
            testing_info = json.loads(f.read(), encoding='utf-8')

        self._compare_infos(stable_info, testing_info)

    def _compare_infos(self, stable_info, testing_info):
        pass

    def _make_info_subtask(self, state):
        return self.create_subtask(
            task_type='VIDEO_BUILD_INDEX_INFO',
            description='Prepare index info for state: {}'.format(state),
            input_parameters={'index_state': state},
        )

    def _get_resource(self, state):
        return apihelpers.get_last_resource_with_attrs(
            resource_types.VIDEO_INDEX_INFO,
            attrs={'state': state},
            all_attrs=True,
        )


__Task__ = VideoIndexValidation
