# -*- coding: utf-8 -*-

import itertools
import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects import resource_types
from sandbox.projects.PatchPlan import PatchPlan
from sandbox.projects.PatchPlan import PlanParameter
from sandbox.projects.PatchPlan import AddCgiParameter
from sandbox.projects.PatchPlan import ResourceAttrsParameter

from sandbox.projects.VideoLoadBasesearchResources.task import VideoLoadBasesearchResources
from sandbox.projects.VideoLoadBasesearchResources.task import MinSerpSearchQueriesParameter
from sandbox.projects.VideoLoadBasesearchResources.task import MinRelatedSearchQueriesParameter
from sandbox.projects.VideoLoadBasesearchResources.task import MinSerpFactorsQueriesParameter
from sandbox.projects.VideoLoadBasesearchResources.task import MinRelatedFactorsQueriesParameter
from sandbox.projects.VideoLoadBasesearchResources.task import MinVideotopQueriesParameter
from sandbox.projects.VideoLoadBasesearchResources.task import MinVideohubQueriesParameter
from sandbox.projects.VideoLoadBasesearchResources.task import MinModelshubQueriesParameter
from sandbox.projects.VideoLoadBasesearchResources.task import SerpSearchQueriesAttrsParameter
from sandbox.projects.VideoLoadBasesearchResources.task import RelatedSearchQueriesAttrsParameter
from sandbox.projects.VideoLoadBasesearchResources.task import SerpFactorsQueriesAttrsParameter
from sandbox.projects.VideoLoadBasesearchResources.task import RelatedFactorsQueriesAttrsParameter
from sandbox.projects.VideoLoadBasesearchResources.task import VideotopQueriesAttrsParameter
from sandbox.projects.VideoLoadBasesearchResources.task import VideohubQueriesAttrsParameter
from sandbox.projects.VideoLoadBasesearchResources.task import ModelshubQueriesAttrsParameter
from sandbox.projects.GetVideoSearchDatabase import GetVideoSearchDatabase
from sandbox.projects.GetVideoSearchDatabase import GetTier0Parameter
from sandbox.projects.GetVideoSearchDatabase import ShardIdParameter
from sandbox.projects.common.BaseLoadMediaBasesearchResources import IndexTypeParameter
from sandbox.projects.common.BaseLoadMediaBasesearchResources import RegionParameter
from sandbox.projects.common.BaseLoadMediaBasesearchResources import MinSnippetsQueriesParameter

from sandbox.projects.common.BaseLoadMediaBasesearchResources import AllQueriesAttrsParameter
from sandbox.projects.common.BaseLoadMediaBasesearchResources import SnippetsQueriesAttrsParameter

from sandbox.projects.common.search.settings import VideoSettings
from sandbox.projects.common.search.settings import INDEX_MAIN
from sandbox.projects.common.search.settings import VLA_REGION
from sandbox.projects.common.search.settings import SHARD_INSTANCE_ATTRIBUTE_NAME

from sandbox.projects.common.utils import check_if_tasks_are_ok

from sandbox.projects.common import apihelpers
# from projects.common import cms
from sandbox.projects.common import utils


_PRIEMKA_ATTRIBUTES = VideoSettings.RESOURCE_PRIEMKA_ATTRIBUTES
_SERP_SEARCH_ATTRIBUTES = VideoSettings.RESOURCE_SERP_SEARCH_ATTRIBUTES
_SERP_FACTORS_ATTRIBUTES = VideoSettings.RESOURCE_SERP_FACTORS_ATTRIBUTES
_RELATED_SEARCH_ATTRIBUTES = VideoSettings.RESOURCE_RELATED_SEARCH_ATTRIBUTES
_RELATED_FACTORS_ATTRIBUTES = VideoSettings.RESOURCE_RELATED_FACTORS_ATTRIBUTES
_SNIPPETS_ATTRIBUTES = VideoSettings.RESOURCE_SNIPPETS_ATTRIBUTES
_VIDEOTOP_SEARCH_ATTRIBUTES = VideoSettings.RESOURCE_VIDEOTOP_SEARCH_ATTRIBUTES
_VIDEOHUB_SEARCH_ATTRIBUTES = VideoSettings.RESOURCE_VIDEOHUB_SEARCH_ATTRIBUTES
_MODELSHUB_FETCH_DOC_ATTRIBUTES = VideoSettings.RESOURCE_MODELSHUB_FETCH_DOC_ATTRIBUTES

_INDICES = VideoSettings.INDICES_FOR_TESTENV
_REGIONS = (VLA_REGION,)
_PRIEMKA_SOURCE = VideoSettings.PRIEMKA_SOURCE

_HEAVY_QUERY_PARAMETERS = '&pron=noprune&pron=nofastrank'
"""
    Параметры для 'утяжеления' запросов
"""

_HEATUP_QUERY_PARAMETERS = '&ignoreGen=da'
"""
    Параметры для грелки
"""

_LOAD_TASK_IDS = 'load_task_ids'
"""
    Идентификаторы задачи по загрузке данных
"""

_PATCH_TASK_IDS = 'patch_task_ids'
"""
    Идентификатор задачи по созданию "тяжёлых" запросов
"""

_LOAD_DATABASE_IDS = "load_base_database_ids"


def _make_attributes(attributes):
    """
        Формирует список атрибутов для последующего присваивания
    """

    return ','.join('%s=%s' % a for a in attributes)


def _get_task_resource(task_ids, resource_type, attribute_name, attribute_value):
    """
        Возвращает ресурс указанной задачи с указанными атрибутами
    """

    for task_id in task_ids:
        resources = apihelpers.list_task_resources(task_id, resource_type=resource_type)
        for resource in resources:
            if resource.attributes.get(attribute_name) == attribute_value:
                return resource

    raise SandboxTaskFailureError(
        "Cannot find resource {} in task {} with attribute {}={}".format(
            resource_type,
            task_id,
            attribute_name,
            attribute_value,
        )
    )


class VideoLoadBasesearchTestenvData(SandboxTask):
    """
        Задача загрузки актуальных данных с production
        для проведения тестирования и приёмки
    """

    type = 'VIDEO_LOAD_BASESEARCH_TESTENV_DATA'

    input_parameters = (
        GetTier0Parameter,
        ShardIdParameter,
        MinSerpSearchQueriesParameter,
        MinRelatedSearchQueriesParameter,
        MinSerpFactorsQueriesParameter,
        MinRelatedFactorsQueriesParameter,
        MinSnippetsQueriesParameter,
        MinVideotopQueriesParameter,
        MinVideohubQueriesParameter,
        MinModelshubQueriesParameter,
    )

    def on_execute(self):
        if _LOAD_TASK_IDS not in self.ctx:
            self.ctx[_LOAD_TASK_IDS] = [
                self._load_queries(index_type, region)
                for index_type, region in itertools.product(_INDICES, _REGIONS)
            ]

        if _LOAD_DATABASE_IDS not in self.ctx:
            self.ctx[_LOAD_DATABASE_IDS] = [
                self._load_databases(index_type) for index_type in _INDICES
            ]

        load_task_ids = self.ctx[_LOAD_TASK_IDS]
        utils.check_subtasks_fails(fail_on_first_failure=True)

        if _PATCH_TASK_IDS not in self.ctx:
            check_if_tasks_are_ok(load_task_ids)
            self.ctx[_PATCH_TASK_IDS] = [
                self._patch_heavy_plan()
            ] + [
                self._patch_heatup_plan(index_type) for index_type in _INDICES
            ]

        utils.check_subtasks_fails(fail_on_first_failure=True)

    def _load_databases(self, index_type):
        """
            Запустить подзадачу для скачивания базы с production
        """

        # region = VLA_REGION
        # cms_configuration, instance_tag_name = VideoSettings.basesearch_tag(index_type, region, VideoSettings.get_nanny_token(self))
        # shard_names = cms.get_cms_shards(
        #    instance_tag_name=instance_tag_name,
        #    cms_configuration=cms_configuration
        # )

        nany_client = VideoSettings.get_nanny(VideoSettings.get_nanny_token(self))
        service_name = VideoSettings.SBASE_NANNY_SERVICE_NAME if index_type == INDEX_MAIN else VideoSettings.UBASE_NANNY_SERVICE_NAME

        instances = nany_client.get_service_current_instances(service_name)
        logging.debug('current instances:  %s' % instances)
        instances = instances['result']
        shard_names = [
            meta['hostname'] for meta in instances
        ]

        sub_ctx = {
            'database_shard_name': shard_names[0],
            'resource_attrs': 'video_priemka=1,video_testenv_base_prod_database=yes',
            GetTier0Parameter.name: self.ctx[GetTier0Parameter.name],

            'kill_timeout': 60 * 60 * 10  # 10 hours (Web timeout * 2)
        }
        sub_task = self.create_subtask(
            task_type=GetVideoSearchDatabase.type,
            input_parameters=sub_ctx,
            description="{}, {}".format(self.descr, index_type)
        )
        return sub_task.id

    def _load_queries(self, index_type, region):
        """
            Запустить подзадачу для скачивания данных с production
        """

        all_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'all')]
        serp_search_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'serp_search')]
        serp_factors_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'serp_factors')]
        related_search_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'related_search')]
        related_factors_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'related_factors')]
        snippets_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'snippets')]
        videotop_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'videotop_search')]
        videohub_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'videohub_search')]
        modelshub_fetch_doc_attributes = [VideoSettings.testenv_basesearch_queries_attributes(index_type, region, 'video_modelshub_fetch_doc')]

        if (index_type, region) == _PRIEMKA_SOURCE:
            serp_search_attributes.append(_SERP_SEARCH_ATTRIBUTES)
            serp_factors_attributes.append(_SERP_FACTORS_ATTRIBUTES)
            related_search_attributes.append(_RELATED_SEARCH_ATTRIBUTES)
            related_factors_attributes.append(_RELATED_FACTORS_ATTRIBUTES)
            snippets_attributes.append(_SNIPPETS_ATTRIBUTES)
            videotop_attributes.append(_VIDEOTOP_SEARCH_ATTRIBUTES)
            videohub_attributes.append(_VIDEOHUB_SEARCH_ATTRIBUTES)
            modelshub_fetch_doc_attributes.append(_MODELSHUB_FETCH_DOC_ATTRIBUTES)

        sub_ctx = {
            IndexTypeParameter.name: index_type,
            RegionParameter.name: region,

            GetTier0Parameter.name: self.ctx[GetTier0Parameter.name],
            ShardIdParameter.name: self.ctx[ShardIdParameter.name],
            MinSerpSearchQueriesParameter.name: self.ctx[MinSerpSearchQueriesParameter.name],
            MinRelatedSearchQueriesParameter.name: self.ctx[MinRelatedSearchQueriesParameter.name],
            MinSerpFactorsQueriesParameter.name: self.ctx[MinSerpFactorsQueriesParameter.name],
            MinRelatedFactorsQueriesParameter.name: self.ctx[MinRelatedFactorsQueriesParameter.name],
            MinSnippetsQueriesParameter.name: self.ctx[MinSnippetsQueriesParameter.name],
            MinVideotopQueriesParameter.name: self.ctx[MinVideotopQueriesParameter.name],
            MinVideohubQueriesParameter.name: self.ctx[MinVideohubQueriesParameter.name],
            MinModelshubQueriesParameter.name: self.ctx[MinModelshubQueriesParameter.name],

            AllQueriesAttrsParameter.name: _make_attributes(all_attributes),
            SerpSearchQueriesAttrsParameter.name: _make_attributes(serp_search_attributes),
            SerpFactorsQueriesAttrsParameter.name: _make_attributes(serp_factors_attributes),
            RelatedSearchQueriesAttrsParameter.name: _make_attributes(related_search_attributes),
            RelatedFactorsQueriesAttrsParameter.name: _make_attributes(related_factors_attributes),
            SnippetsQueriesAttrsParameter.name: _make_attributes(snippets_attributes),
            VideotopQueriesAttrsParameter.name: _make_attributes(videotop_attributes),
            VideohubQueriesAttrsParameter.name: _make_attributes(videohub_attributes),
            ModelshubQueriesAttrsParameter.name: _make_attributes(modelshub_fetch_doc_attributes),

            'kill_timeout': 60 * 60 * 48  # 48 hours
        }
        sub_task = self.create_subtask(
            task_type=VideoLoadBasesearchResources.type,
            input_parameters=sub_ctx,
            description="{}, {} {}".format(self.descr, index_type, region)
        )
        return sub_task.id

    def _patch_heavy_plan(self):
        """
            Запустить подзадачу для создания тяжёлых запросов
        """

        orig_attributes = VideoSettings.testenv_basesearch_queries_attributes(
            _PRIEMKA_SOURCE[0], _PRIEMKA_SOURCE[1], 'all')
        new_attributes = VideoSettings.testenv_basesearch_queries_attributes(
            _PRIEMKA_SOURCE[0], _PRIEMKA_SOURCE[1], 'heavy')

        return self._patch_plan(
            orig_attributes, new_attributes, _HEAVY_QUERY_PARAMETERS, 'heavy')

    def _patch_heatup_plan(self, index_type):
        """
            Запустить подзадачу для создания прогревочных запросов
        """

        orig_attributes = VideoSettings.testenv_basesearch_queries_attributes(
            index_type, VideoSettings.HEATUP_REGION, 'all')
        new_attributes = VideoSettings.heatup_basesearch_attributes(index_type)

        return self._patch_plan(
            orig_attributes, new_attributes, _HEATUP_QUERY_PARAMETERS, 'heatup')

    def _patch_plan(self, orig_attributes, new_attributes, cgi_parameters, plan_description):
        """
            Запустить подзадачу для модифицирования запросов
        """

        load_task_ids = self.ctx[_LOAD_TASK_IDS]

        plan_resource = _get_task_resource(
            load_task_ids,
            resource_types.BASESEARCH_PLAN,
            orig_attributes[0],
            orig_attributes[1]
        )
        shard_instance_attribute = SHARD_INSTANCE_ATTRIBUTE_NAME

        plan_attributes = [(shard_instance_attribute, plan_resource.attributes[shard_instance_attribute])]
        plan_attributes.append(new_attributes)

        sub_ctx = {
            PlanParameter.name: plan_resource.id,
            AddCgiParameter.name: cgi_parameters,
            ResourceAttrsParameter.name: _make_attributes(plan_attributes)
        }
        sub_task = self.create_subtask(
            task_type=PatchPlan.type,
            input_parameters=sub_ctx,
            description="{}, {}".format(self.descr, plan_description),
        )

        return sub_task.id


__Task__ = VideoLoadBasesearchTestenvData
