# coding: utf-8

import datetime
import logging
import time

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2


_DAYS = 30
_REPORT_NAME = 'ExtData/YandexVideoMetrics/VideoContentSystem/FullIndexStability'
_REPORT_SCALE = 'hourly'
_RELEASE_TABLE = '//home/videoindex/full/index/deployed/production'


class VideoMetricsIndexStability(SandboxTask):
    """
        Расчет стабильности переключения поисковых баз видео
    """
    type = 'VIDEO_METRICS_INDEX_STABILITY'

    environment = (
        PipEnvironment('python-statface-client', '0.14.0', use_wheel=True),
        PipEnvironment("yandex-yt"),
    )

    def on_execute(self):
        import yt.wrapper as yt

        yt_client = yt.YtClient(proxy="arnold.yt.yandex.net", token=sdk2.Vault.data('VIDEODEV', 'yt_token'))

        now = datetime.datetime.now()
        sliding_window = now - datetime.timedelta(days=_DAYS)
        releases = list(yt_client.select_rows("* from [{}] where ts > {}".format(_RELEASE_TABLE, int(time.mktime(sliding_window.timetuple()))), format=yt.JsonFormat()))

        self.set_info('Releases number for the last {} days: {}'.format(_DAYS, len(releases)), do_escape=False)
        stability = float(len(releases))/_DAYS * 100
        self.set_info('Stability metric is: {0:.2f}%'.format(stability))
        logging.debug('Here are releases for the last {} days'.format(_DAYS))

        self.statfaceUpload(stability)

    def statfaceUpload(self, stability):
        from statface_client import StatfaceClient
        stat_user = "robot_robot-video-acc"
        stat_pass = self.get_vault_data('robot-video-acc_statface')

        stat = StatfaceClient(stat_user, stat_pass, 'upload.stat.yandex-team.ru')
        report = stat.get_report(_REPORT_NAME)
        data = dict()
        data['stability'] = stability
        data['fielddate'] = self._get_date(_REPORT_SCALE)

        logging.debug('upload data: {}'.format(data))
        report.upload_data(_REPORT_SCALE, [data])

    def _get_date(self, scale):
        fmt = {
            'daily': '%Y-%m-%d',
            'hourly': '%Y-%m-%d %H:00:00',
            'minutely': '%Y-%m-%d %H:%M:00',
        }.get(scale, None)
        if fmt is None:
            logging.fatal('Invalid scale value')
        return datetime.datetime.now().strftime(fmt)


__Task__ = VideoMetricsIndexStability
