# coding: utf-8

import datetime
import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects import resource_types
import json

_REPORT_NAME = 'ExtData/YandexVideoMetrics/VideoContentSystem/IndexShardSize'
_REPORT_SCALE = 'hourly'


class VideoMetricsShardSize(SandboxTask):
    """
        Index shard size tool
    """
    type = 'VIDEO_METRICS_SHARD_SIZE'

    environment = (
        PipEnvironment('python-statface-client', '0.14.0', use_wheel=True),
    )

    def on_execute(self):
        released = channel.sandbox.list_releases(
            resource_type=resource_types.VIDEO_INDEX_INFO, limit=1)[0]

        self.set_info('Last released task id: {}'.format(released.task_id))

        task = channel.sandbox.get_task(released.task_id)
        info = json.loads(task.info[task.info.index('{'):])
        shardsize_platinum = info['VideoPlatinum']['shards']['avg_size_gb']
        self.set_info('Platinum shard size is: {0:.2f}'.format(shardsize_platinum))
        shardsize_tier0 = info['VideoTier0']['shards']['avg_size_gb']
        self.set_info('Tier0 shard size is: {0:.2f}'.format(shardsize_tier0))

        indexsize = info['index_size']
        self.set_info('Number of docs in index is {}'.format(indexsize))
        docsize_platinum = info['VideoPlatinum']['avg_doc_size']
        docsize_tier0 = info['VideoTier0']['avg_doc_size']
        self.set_info('Avg doc size in platinum is {} and {} in tier0'.format(docsize_platinum, docsize_tier0))
        platinum_docs = info['platinum_docs']
        tier0_docs = info['tier0_docs']
        self.set_info('Doc count in platinum is {} and {} in tier0'.format(platinum_docs, tier0_docs))

        self.statfaceUpload(shardsize_platinum, shardsize_tier0, indexsize, docsize_platinum, docsize_tier0, platinum_docs, tier0_docs)

    def statfaceUpload(self, shardsize_platinum, shardsize_tier0, indexsize, docsize_platinum, docsize_tier0, platinum_docs, tier0_docs):
        from statface_client import StatfaceClient
        stat_user = "robot_robot-video-acc"
        stat_pass = self.get_vault_data('robot-video-acc_statface')

        stat = StatfaceClient(stat_user, stat_pass, 'upload.stat.yandex-team.ru')
        report = stat.get_report(_REPORT_NAME)
        data = dict()
        data['shardsizeplatinum'] = shardsize_platinum
        data['shardsizetier0'] = shardsize_tier0
        data['indexsize'] = indexsize
        data['fielddate'] = self._get_date(_REPORT_SCALE)
        data['docsizeplatinum'] = docsize_platinum
        data['docsizetier0'] = docsize_tier0
        data['platinum_docs'] = platinum_docs
        data['tier0_docs'] = tier0_docs

        logging.debug('upload data: {}'.format(data))
        report.upload_data(_REPORT_SCALE, [data])

    def _get_date(self, scale):
        fmt = {
            'daily': '%Y-%m-%d',
            'hourly': '%Y-%m-%d %H:00:00',
            'minutely': '%Y-%m-%d %H:%M:00',
        }.get(scale, None)
        if fmt is None:
            logging.fatal('Invalid scale value')
        return datetime.datetime.now().strftime(fmt)


__Task__ = VideoMetricsShardSize
