# coding: utf-8

import datetime
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
import sandbox.projects.MediaLib.VideoMetricsStability as vms
from sandbox.projects.common.juggler import jclient


class QueueDepth(SandboxIntegerParameter):
    name = 'queue_depth'
    description = 'Queue depth'
    required = True


_STATFACE_REPORT_NAME = 'ExtData/YandexVideoMetrics/VideoContentSystem/UltraThumbsRecrawlDepth'
_STATFACE_REPORT_SCALE = 'minutely'

_JUGGLER_HOST = 'video-robot'
_JUGGLER_SERVICE = 'thdb_force_ultra_recrawl_depth'

_QUEUE_THRESHOLD = 10000


class VideoMetricsUltraRecrawlDepth(vms.VideoMetricsStability):
    """
        Метрика глубины очереди непрокачанных ультра тумбов
    """

    type = 'VIDEO_METRICS_ULTRA_RECRAWL_DEPTH'

    cores = 1
    execution_space = 1024  # 1 Gb

    input_parameters = [
        QueueDepth
    ]

    def notify_juggler(self, status, msg):
        try:
            jclient.send_events_to_juggler(
                _JUGGLER_HOST,
                _JUGGLER_SERVICE,
                status,
                msg
            )
        except Exception:
            self.set_info('Failed to send event to juggler')

    def get_status(self, queue):
        if queue > _QUEUE_THRESHOLD:
            return 'CRIT'
        return 'OK'

    def on_execute(self):
        now = datetime.datetime.now()
        data = {}
        data['fielddate'] = self._scale_date(
            now,
            _STATFACE_REPORT_SCALE
        )
        queue_depth = self.ctx[QueueDepth.name]
        data['queue'] = queue_depth
        self.statface_upload(
            _STATFACE_REPORT_NAME,
            _STATFACE_REPORT_SCALE,
            [data]
        )
        msg = 'Queue depth: {}'.format(queue_depth)
        self.notify_juggler(
            self.get_status(queue_depth),
            msg
        )


__Task__ = VideoMetricsUltraRecrawlDepth
