import os

from sandbox.sandboxsdk import process

from sandbox.projects import resource_types
from sandbox.projects.common.mediasearch import ban as mediaban
from sandbox.projects.common.nanny import auto_deploy
from sandbox.sandboxsdk.svn import Arcadia


def format_attrs_list(line):
    fields = line.strip().split()
    return "#query\t%s\t*\t0\t%s\n" % (fields[0], '\t'.join(fields[1:]))


class VideoReleaseContentTries(auto_deploy.AutoNannyDeployTask, mediaban.VideoBaseReleaseBanTask):
    """
        Builds content attributes trie for Yandex.Video service
    """

    type = "VIDEO_RELEASE_CONTENT_ATTRS"

    release_subject = "video/middle/content-data-{timestamp}"
    release_comment = "video content attrs release"
    release_resources = (
        resource_types.VIDEO_MIDDLESEARCH_CONTENT_ATTRS,
        resource_types.VIDEO_MIDDLESEARCH_CONTENT_ATTRS_PLAIN,
    )

    def get_stable_services(self):
        return [self._SERVICE_ID]

    def get_nanny_oauth_token(self):
        return self.get_vault_data('VIDEO-ROBOT', 'robot-video-crawl-nanny-oauth')

    def _build_ban(self):
        content_attrs_path = self.abs_path(resource_types.VIDEO_MIDDLESEARCH_CONTENT_ATTRS_PLAIN.basename)
        svn_content_attrs_path = content_attrs_path + ".svn"
        svn_content_attrs_arcadia_url = "arcadia:/arc/trunk/arcadia/yweb/webscripts/video/frames/content_attrs"
        Arcadia.export(svn_content_attrs_arcadia_url, svn_content_attrs_path)

        self._format_file(svn_content_attrs_path, content_attrs_path, format_func=format_attrs_list)

        replace_attrs_path = content_attrs_path + ".svn.rep"
        svn_replace_attrs_path = "arcadia:/arc/trunk/arcadia/yweb/webscripts/video/index/config/replace_attrs"
        Arcadia.export(svn_replace_attrs_path, replace_attrs_path)

        self._format_file(replace_attrs_path, content_attrs_path, format_func=format_attrs_list, append=True)

        has_changes = self._update_resource(resource_types.VIDEO_MIDDLESEARCH_CONTENT_ATTRS_PLAIN, path=content_attrs_path)

        if not has_changes:
            return 0

        content_attrs_trie_path = self.abs_path(resource_types.VIDEO_MIDDLESEARCH_CONTENT_ATTRS.basename)
        indexer_tool = self._tool(resource_types.VIDEO_QUERYDATAINDEXER_EXECUTABLE)
        process.run_process([
            indexer_tool,
            "-S", "contentattrs",
            "-N", "exacturl,tld,ipregregion",
            "-i", content_attrs_path,
            "-o", content_attrs_trie_path], outputs_to_one_file=False, log_prefix="indexer")

        viewer_tool = self._tool(resource_types.VIDEO_QUERYDATAVIEWER_EXECUTABLE)
        process.run_process([
            viewer_tool,
            "-H",
            "-i", content_attrs_trie_path], outputs_to_one_file=False, log_prefix="viewer")

        self.create_resource(self.descr, content_attrs_trie_path, resource_types.VIDEO_MIDDLESEARCH_CONTENT_ATTRS)

        return os.stat(content_attrs_trie_path).st_size


__Task__ = VideoReleaseContentTries
