import logging

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process

from sandbox.projects import resource_types
from sandbox.projects.common.mediasearch import ban as mediaban
from sandbox.projects.common.nanny import auto_deploy


class YTServerParameter(parameters.SandboxStringParameter):
    name = "yt_server"
    description = "YT server"
    required = True


def cut_https_scheme(url):
    if url.startswith('https://'):
        return url[8:]
    return url


def cut_last_slash(url):
    if url.endswith("/"):
        return url[:-1]
    return url


def format_soft_404_ban(line):
    url, status = line.strip().split('\t', 2)
    # if status == "removed":
    #    return None
    if url.find('/') == len(url) - 1:
        return None
    return "%s\n" % (cut_https_scheme(url))


def format_grimhold_ban(line):
    url, tld, region, bantype, _, sourceban, _, ban_stub_url1, ban_stub_url2 = line.rstrip("\n").split("\t")
    ban_stub_url = ban_stub_url1 if ban_stub_url1 else ban_stub_url2
    url_l = url.lower()
    if url == 'http://' or url_l.endswith('.jpg') or url_l.endswith('.jpeg') or url_l.endswith('.gif'):
        return None
    if sourceban == "common":
        sourceban = "manban"
    if region == "*":
        region = "0"
    bantype = bantype.replace("bantype=", "")
    if tld == '*' and region == '0' and bantype == 'url' and sourceban == 'manban' and not ban_stub_url:
        return url + '\n'
    else:
        return ''


def format_url_ban(line):
    return line


class VideoReleaseVideobanHashesBase(auto_deploy.AutoNannyDeployTask, mediaban.VideoBaseReleaseBanTask):
    """
        Base class for building videoban.soft404.vec for Yandex.Video service
    """

    def get_stable_services(self):
        return [self._SERVICE_ID]

    def get_nanny_oauth_token(self):
        return self.get_vault_data('VIDEO-ROBOT', 'robot-video-crawl-nanny-oauth')

    def _build_ban(self):
        NotImplemented

    def _read_yt_table(self, table_path, dst_path, append, format_func):
        yt_exists = self._yt("exists", table_path)
        if yt_exists.wait():
            raise errors.SandboxTaskFailureError("Failed to check existence %s" % (table_path))

        exists = False
        with open(yt_exists.stdout_path, 'r') as f:
            exists = str.strip(f.readline()) == "true"

        if exists:
            yt_read = self._yt("read", "--format", "yamr", table_path)
            if yt_read.wait():
                raise errors.SandboxTaskFailureError("Failed to read %s" % (table_path))
            self._format_file(yt_read.stdout_path, dst_path, format_func=format_func, append=append)

    def _yt(self, cmd, *args):
        yt_tool = self._tool(resource_types.VIDEO_YT_PYTHON_EXECUTABLE)
        yt_args = (
            yt_tool,
            "--proxy", self.ctx[YTServerParameter.name],
        )

        return process.run_process(
            yt_args + (cmd,) + args,
            environment={"YT_TOKEN": self.get_vault_data('VIDEODEV', 'yt_token')},
            outputs_to_one_file=False,
            log_prefix="yt.{}".format(cmd),
            timeout=600,
            wait=False
        )

    def _update_service(self, service_id):
        logging.info("_update_service: {}".format(str(service_id)))
        auto_deploy.AutoNannyDeployTask._update_service(self, service_id)

    def _sort_file(self, in_filename, out_filename):
        with open(in_filename, 'r') as in_file, open(out_filename, 'w') as out_file:
            lines = in_file.readlines()
            lines.sort()
            out_file.writelines(lines)

    def _run_url2fastban_tool(self, input, output, log_prefix):
        url2fastban_tool = self._tool(resource_types.VIDEO_URL2FASTBAN_EXECUTABLE)
        process.run_process([
            url2fastban_tool,
            "makeHashes",
            "-i", input,
            "-o", output], outputs_to_one_file=False, log_prefix=log_prefix)
