from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskStop

import logging

import sandbox.projects.common.constants as consts

from sandbox.projects.VideoSearch.VideoBuildDocbaseBundle import VideoBuildDocbaseBundle
from sandbox.projects.VideoSearch.VideoBuildIndexBundle import VideoBuildIndexBundle


class VideoBuildArtifacts(sdk2.Task):
    """
        Build artifacts after each commit
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1 * 1024  # 1 gb
        ram = 512

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia")
        revision = sdk2.parameters.String("Revision", default="", required=True)

    def run_tasks(self, run_id, func):
        if not self.Context.tasks_waiting_for:
            self.Context.tasks_waiting_for = {}

        if run_id in self.Context.tasks_waiting_for:
            tasks = self.find(id=self.Context.tasks_waiting_for[run_id])

            if not all([task.status in ctt.Status.Group.SUCCEED for task in tasks]):
                raise TaskStop('Failed task dependencies, could not continue')

            return tasks

        tasks = func()
        self.Context.tasks_waiting_for[run_id] = [task.id for task in tasks]
        logging.info('Will wait for task dependencies: {}'.format(self.Context.tasks_waiting_for[run_id]))
        raise sdk2.WaitTask([task.enqueue() for task in tasks], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, True)

    def build_webscripts(self):
        return (
            sdk2.Task["VIDEO_BUILD_WEBSCRIPTS"](
                self,
                arch='linux',
                description="docbase {}".format(self.Parameters.arcadia_url),
            ),
        )

    def build_binaries(self, webscripts_task):
        return (
            sdk2.Task["VIDEO_BUILD_DOCBASE_BINARIES"](
                self,
                arch='linux',
                description="docbase {}".format(self.Parameters.arcadia_url),
                **{consts.ARCADIA_URL_KEY: self.Parameters.arcadia_url}
            ),
        )

    def build_bundles(self, binaries_task):
        return [bundle_task_type(
                    self,
                    description="docbase {}".format(self.Parameters.arcadia_url),
                    BuildTask=binaries_task.id,
                )
            for bundle_task_type in VideoBuildDocbaseBundle, VideoBuildIndexBundle]

    def on_execute(self):
        self.run_tasks(
            "bundles",
            lambda: self.build_bundles(
                self.run_tasks(
                    "binaries",
                    lambda: self.build_binaries(
                        self.run_tasks(
                            "webscripts",
                            lambda: self.build_webscripts()
                        )
                    )
                ).first()
            )
        )
