# -*- coding: utf-8 -*-_read_resource

import logging
import json
from datetime import datetime as dt

from fileinfo import VideoIndexFileInfo

from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common.ReleasedResourceConsumer import ReleasedResourceConsumer

from sandbox.projects.common.solomon import push_to_solomon_v2


class VideoIndexStateParameter(parameters.SandboxStringParameter):
    name = "index_state"
    description = "Index state"
    required = True


class VideoBuildIndexInfo(task.SandboxTask, ReleasedResourceConsumer):
    """
        Builds video index info for Yandex.Video service
    """

    type = 'VIDEO_BUILD_INDEX_INFO'

    input_parameters = (
        VideoIndexStateParameter,
    )

    def on_execute(self):
        solomon_token = self.get_vault_data('VIDEO-ROBOT', 'robot-video-solomon-token')
        indexstate = self.ctx[VideoIndexStateParameter.name]

        fileinfo = json.loads(VideoIndexFileInfo(self, indexstate).__str__())
        logging.info(fileinfo)

        params = {
            "project": "video",
            "cluster": "video_robot",
            "service": "index_info",
        }

        ts = int(dt.strptime(indexstate, "%Y%m%d-%H%M.idx").strftime('%s'))

        sensors = [
            {"labels": {"sensor": "index_size"}, "value": fileinfo["index_size"], "ts": ts},
            {"labels": {"sensor": "platinum_docs"}, "value": fileinfo["platinum_docs"], "ts": ts},
            {"labels": {"sensor": "tier0_docs"}, "value": fileinfo["tier0_docs"], "ts": ts},
            {"labels": {"sensor": "player_count"}, "value": fileinfo["player_count"], "ts": ts},
        ]

        for shard in fileinfo["shards"]:
            for stat in ["min", "avg", "max"]:
                sensors.append({"labels": {"sensor": "shard_size", "shard": str(shard), "statistic": stat}, "value": fileinfo["shards"][shard]["common"]["shards"][stat + "_size_gb"], "ts": ts})

        logging.info(sensors)

        push_to_solomon_v2(solomon_token, params, sensors)

        out_info = open(self.abs_path('info.txt'), 'w')
        out_info.write(json.dumps(fileinfo, indent=4))

        resource_id = self.create_resource(
            "video index info",
            'state_{}'.format(self.ctx[VideoIndexStateParameter.name]),
            resource_types.VIDEO_INDEX_INFO
        ).id

        resource = channel.sandbox.get_resource(resource_id)
        paths.copy_path(self.abs_path('info.txt'), resource.path)

        channel.task.mark_resource_ready(resource)


__Task__ = VideoBuildIndexInfo
