# -*- coding: utf-8 -*-_read_resource

import logging
import json
from json import encoder
from datetime import datetime as dt
from sandbox.common.share import files_torrent

from sandbox.projects import resource_types
from sandbox.projects.MediaLib.iss_shardtracker import iss_shardtracker_api

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import process

_PLATINUM_GROUPS_COUNT = 12
_PLATINUM_SHARDS_IN_GROUP = 18
_PLATINUM_SHARDS_COUNT = _PLATINUM_GROUPS_COUNT * _PLATINUM_SHARDS_IN_GROUP

_TIER0_GROUPS_COUNT = 14
_TIER0_SHARDS_IN_GROUP = 18
_TIER0_SHARDS_COUNT = _TIER0_GROUPS_COUNT * _TIER0_SHARDS_IN_GROUP

class VideoIndexFileInfo:
    def __init__(self, parent_task, indexstate):
        self.parent = parent_task
        self.indexstate = indexstate

    def __str__(self):
        state = dt.strptime(self.indexstate, "%Y%m%d-%H%M.idx").strftime("%Y%m%d-%H%M%S")

        def generate_flat_shard_list(shard_prefix, count):
            return ["{shard_prefix}-{shard_id:03d}-{state}".format(shard_id=i, state=state, shard_prefix=shard_prefix) for i in range(count)]

        def generate_grouped_shard_list(shard_prefix, groups_count, shards_in_group):
            return ["{shard_prefix}-{group_id:03d}-{shard_in_group_id:03d}-{state}".format(group_id=i, shard_in_group_id=j, state=state, shard_prefix=shard_prefix) for j in range (shards_in_group) for i in range(groups_count)]

        info = {
            "shards": {
                "VideoPlatinum": self._get_shard_info(generate_flat_shard_list('vidsidxpt', _PLATINUM_SHARDS_COUNT)),
                "VideoEmbeddingPlatinum": self._get_shard_info(generate_flat_shard_list('VideoEmbeddingPlatinum', _PLATINUM_GROUPS_COUNT)),
                "VideoInvertedPlatinum": self._get_shard_info(generate_grouped_shard_list('VideoInvertedPlatinum', _PLATINUM_GROUPS_COUNT, _PLATINUM_SHARDS_IN_GROUP)),
                "VideoTier0": self._get_shard_info(generate_flat_shard_list('vidsidx', _TIER0_SHARDS_COUNT)),
                "VideoEmbeddingTier0": self._get_shard_info(generate_flat_shard_list('VideoEmbeddingTier0', _TIER0_GROUPS_COUNT)),
                "VideoInvertedTier0": self._get_shard_info(generate_grouped_shard_list('VideoInvertedTier0', _TIER0_GROUPS_COUNT, _TIER0_SHARDS_IN_GROUP)),
            },
            "index_size": self._get_index_size(),
            "platinum_docs": self._get_platinum_docs(),
            "tier0_docs": self._get_tier0_docs(),
            "player_count": self._get_player_count(),
        }

        self._log_report(info)
        logging.debug('collecting shards info complete')
        return json.dumps(info, indent=4)

    def _get_shard_info(self, shards):
        shard_infos = []
        with iss_shardtracker_api() as thrift_shardtracker_client:
            for shard_name in shards:
                logging.info("Getting info for shard [{}]".format(shard_name))

                shard_info = thrift_shardtracker_client.getShardInfo(shard_name)
                shard_size = shard_info.shardInfo.full.size

                rbtorrent = next(iter(shard_info.shardInfo.full.urls))
                files_info = [{"name": fi["name"], "size": fi["size"]} for fi in files_torrent(rbtorrent)]

                shard_infos.append({
                    'shardname': shard_name,
                    'size': shard_size,
                    'rbtorrent': rbtorrent,
                    'files': files_info,
                })

        common_info = self._get_common_info(shard_infos)

        info = {
            "shards": shard_infos,
            "common": common_info,
        }
        return info

    def _log_report(self, info):
        """
            generates basic index info report
        """
        encoder.FLOAT_REPR = lambda o: format(o, '.2f')
        log_info = {
            "VideoPlatinum": info['shards']['VideoPlatinum']['common'],
            "VideoInvertedPlatinum": info['shards']['VideoInvertedPlatinum']['common'],
            "VideoEmbeddingPlatinum": info['shards']['VideoEmbeddingPlatinum']['common'],
            "VideoTier0": info['shards']['VideoTier0']['common'],
            "VideoInvertedTier0": info['shards']['VideoInvertedTier0']['common'],
            "VideoEmbeddingTier0": info['shards']['VideoEmbeddingTier0']['common'],
            "index_size": info['index_size'],
            "platinum_docs": info['platinum_docs'],
            "tier0_docs": info['tier0_docs']
        }
        loginfo = "common info:\n{}".format(json.dumps(log_info, indent=4))
        self.parent.set_info(loginfo, do_escape=False)

    def _get_common_info(self, shard_infos):
        """
            collects common data among index shards
        """
        shard_sizes = []
        for shard_info in shard_infos:
            shard_sizes.append(float(shard_info['size']) / 2**30)

        common_shards_info = {
            "min_size_gb": min(shard_sizes),
            "avg_size_gb": sum(shard_sizes)/len(shard_sizes),
            "max_size_gb": max(shard_sizes)
        }

        common_info = {
            "shards": common_shards_info,
        }

        return common_info

    def _get_table_size(self, rel_path):
        table_path = '//home/videoindex/full/index/' + self.indexstate + '/' + rel_path + '/@row_count'
        yt_get = self._yt("--proxy", "arnold", "get", table_path)

        if yt_get.wait():
            raise errors.SandboxTaskFailureError("Failed to get %s" % (table_path))

        with open(yt_get.stdout_path, 'r') as f:
            table_size = str.strip(f.readline())

        return int(table_size)

    def _get_platinum_docs(self):
        return self._get_table_size('index-pt/output/searchable_index')

    def _get_tier0_docs(self):
        return self._get_table_size('index/output/searchable_index')

    def _get_index_size(self):
        return self._get_table_size('index/sbr/searchable_plan')

    def _get_player_count(self):
        return self._get_table_size('index-pt/output/searchable_players') + self._get_table_size('index/output/searchable_players')

    def _yt(self, cmd, *args):
        yt_tool = self.parent.last_released_resource(resource_types.VIDEO_YT_PYTHON_EXECUTABLE, 'yt_python_resource_id')
        yt_args = (
            yt_tool,
        )
        return process.run_process(
            yt_args + (cmd,) + args,
            environment={"YT_TOKEN": self.parent.get_vault_data('VIDEODEV', 'yt_token')},
            outputs_to_one_file=False,
            log_prefix="yt.{}".format(cmd),
            timeout=600,
            wait=False
        )
