import logging
import os

from sandbox import sdk2
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.VideoSearch import video_resource_types as resource_types
from sandbox.projects.common.ya_deploy import release_integration


class VideoBuildWebscripts(sdk2.Task, release_integration.ReleaseToYaDeployTask2):
    """
        Release Webscripts
    """

    __arcadia_webscripts_video_path = 'yweb/webscripts/video'
    __tar_resource_name = 'webscripts.tar.gz'
    __checkout_path = 'video'

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1 * 1024  # 1 gb
        ram = 512

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.String("URL to Arcadia", default=Arcadia.trunk_url(), required=False)

    def get_resource_type(self):
        return resource_types.VideoWebscriptsRawResource

    @property
    def webscripts_video_url(self):
        main_arcadia_url = self.Parameters.arcadia_url if self.Parameters.arcadia_url else Arcadia.trunk_url()
        return '{0}/{1}'.format(main_arcadia_url, self.__arcadia_webscripts_video_path)

    def export_webscripts(self):
        local_dir = os.path.realpath(self.__checkout_path)
        url = self.webscripts_video_url
        Arcadia.export(url, local_dir)

    def on_execute(self):
        self.export_webscripts()
        webscripts_raw_res = sdk2.ResourceData(resource_types.VideoWebscriptsRawResource(
            self, "Raw video webscripts", self.__checkout_path
        ))
        webscripts_raw_res.ready()

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("tar")) as pl:
            sp.Popen("tar -czvf {0} {1}".format(self.__tar_resource_name, self.__checkout_path),
                shell=True,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            ).wait()

        webscripts_res = sdk2.ResourceData(resource_types.VideoWebscriptsResource(
            self, "Video webscripts", self.__tar_resource_name
        ))
        webscripts_res.ready()

    def release_to_platform(self, parameters):
        res_id = sdk2.Resource.find(task=self, resource_type=resource_types.VideoWebscriptsResource).first().id
        res_id_raw = sdk2.Resource.find(task=self, resource_type=resource_types.VideoWebscriptsRawResource).first().id
        resources_to_deploy = [
            {'id': res_id, 'type': resource_types.VideoWebscriptsResource},
            {'id': res_id_raw, 'type': resource_types.VideoWebscriptsRawResource},
        ]

    def on_release(self, parameters):
        self.release_to_platform(parameters)
        release_integration.ReleaseToYaDeployTask2.on_release(self, parameters)

        self._send_release_info_to_email(parameters)
        self.mark_released_resources(parameters["release_status"])
