import re, time
import logging
from sandbox import sdk2
from sandbox.sdk2.service_resources import SandboxTasksBinary


class VideoCleanupTestData(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        data_ttl = sdk2.parameters.Integer('TTL for test data in days', default_value=7)

    @property
    def data_ttl(self):
        return 60 * 60 * 24 * self.Parameters.data_ttl

    @property
    def yt_dir(self):
        return '//home/videoindex/beta'

    def get_old_data_timestamps(self, yt_client):
        import yt.wrapper as yt

        timestamp = int(time.time())
        items = list(yt_client.list(self.yt_dir))

        is_directory = lambda item : yt_client.get_attribute(yt.ypath_join(self.yt_dir, item), 'type') == 'map_node'
        directories = filter(is_directory, items)

        is_old = lambda name : re.match(r'testing\d+', name) and int(name.replace('testing', '')) + self.data_ttl < timestamp
        old_directories = filter(is_old, directories)

        timestamps = map(lambda name : name.replace('testing', ''), old_directories)
        return timestamps

    def remove_upstream(self, upstream_id, namespace_id, balancer_id):
        from sandbox.projects.VideoSearch.awacs_helper import AwacsUpdater

        awacs_updater = AwacsUpdater(namespace_id, balancer_id)
        awacs_updater.remove_upstream(upstream_id)
        awacs_updater.wait_for_remove_upstream(upstream_id)

    def remove_backend(self, backend_id, namespace_id, balancer_id):
        from sandbox.projects.VideoSearch.awacs_helper import AwacsUpdater

        awacs_updater = AwacsUpdater(namespace_id, balancer_id)
        awacs_updater.remove_backend(backend_id)
        awacs_updater.wait_for_remove_backend(backend_id)

    def remove_deploy_unit(self, deploy_stage, deploy_unit_id):
        from sandbox.projects.VideoSearch.deploy_helper import DeployUnitRemover

        remover = DeployUnitRemover(deploy_stage, deploy_unit_id)
        remover.remove()
        remover.wait_for_remove()


    def remove_docbase_data(self, timestamp):
        namespace_id = 'video-robot-cm-test.yandex.net'
        balancer_id = 'video-robot-cm-test.yandex.net_vla'

        upstream_id = 'docbase-test-{0}-video-robot-cm-test_in_yandex-team_ru'.format(timestamp)
        self.remove_upstream(upstream_id, namespace_id, balancer_id)

        backend_id = 'video-robot-cm-video-docbase-test-{0}-backend'.format(timestamp)
        self.remove_backend(backend_id, namespace_id, balancer_id)

        deploy_stage = 'video-robot-cm-video-docbase-test'
        deploy_unit_id = 'video-docbase-{0}'.format(timestamp)
        self.remove_deploy_unit(deploy_stage, deploy_unit_id)

    def remove_index_data(self, timestamp):
        namespace_id = 'video-robot-cm-test.yandex.net'
        balancer_id = 'video-robot-cm-test.yandex.net_vla'

        upstream_id = 'index-test-{0}-video-robot-cm-test_in_yandex-team_ru'.format(timestamp)
        self.remove_upstream(upstream_id, namespace_id, balancer_id)

        backend_id = 'video-robot-cm-video-index-test-{0}-backend'.format(timestamp)
        self.remove_backend(backend_id, namespace_id, balancer_id)

        deploy_stage = 'video-robot-cm-video-index-test'
        deploy_unit_id = 'video-index-{0}'.format(timestamp)
        self.remove_deploy_unit(deploy_stage, deploy_unit_id)

    def on_execute(self):
        import yt.wrapper as yt

        yt_client = yt.YtClient(proxy='arnold.yt.yandex.net', token=sdk2.Vault.data('VIDEODEV', 'yt_token'))
        timestamps = self.get_old_data_timestamps(yt_client)
        for ts in timestamps:
            self.remove_docbase_data(ts)
            self.remove_index_data(ts)

            testing_dir_path = yt.ypath_join(self.yt_dir, 'testing' + ts)
            yt_client.remove(testing_dir_path, recursive=True)

            logging.info('Data was successfully deleted, timestamp : {}'.format(ts))

    def on_save(self):
        self.Requirements.tasks_resource = SandboxTasksBinary.find(attrs={
            'type': 'VIDEO_CLEANUP_TEST_DATA'
        }).first().id
