from sandbox.projects.VideoSearch.platform_helper import ClusterMasterApi
from sandbox.projects.VideoSearch.platform_helper import ClusterMasterResponceStatus

from sandbox.projects.VideoSearch import video_resource_types as resource_types
from sandbox.projects.common.apihelpers import get_last_resource_with_attrs
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox import common

import logging
from sandbox import sdk2
import textwrap
import time


class VideoCreateTestEnv(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        BuildBundleTask = sdk2.parameters.Task(
            'Task where video bundle has been built',
            task_type='VIDEO_BUILD_DOCBASE_BUNDLE',
            required=True
        )
        BuildWebscriptsTask = sdk2.parameters.Task(
            'Task where video webscripts has been build\nLeft empty for get last released resource',
            task_type='VIDEO_BUILD_WEBSCRIPTS',
            required=False
        )

    __STABLE_TESTING_DIRECTORY = '//home/videoindex/beta/testing'
    __CLUSTER_MASTER_API_URL = 'https://video-robot-cm-test.in.yandex-team.ru/docbase/'
    __config_resource_type = resource_types.VideoConfigDocbase
    __webscripts_resource_type = resource_types.VideoWebscriptsRawResource
    __resource_type_to_deploy_resource = {
        'VIDEO_BUNDLE_DOCBASE': 'VIDEO_BUNDLE_DOCBASE',
        'VIDEO_WEBSCRIPTS_RAW_RESOURCE': 'VIDEO_WEBSCRIPTS_RAW',
        'VIDEO_CONFIG_DOCBASE': 'VIDEO_CONFIG_DOCBASE'
    }

    @property
    def cluster_master_api_url(self):
        return '{0}{1}/'.format(self.__CLUSTER_MASTER_API_URL, self.timestamp)

    @property
    def timestamp(self):
        if not self.Context.timestamp:
            self.Context.timestamp = str(int(time.time()))

        return self.Context.timestamp

    @property
    def yt_prefix(self):
        return '//home/videoindex/beta/testing{}/'.format(self.timestamp)

    @sdk2.header()
    def cluster_master_link_report(self):
        return textwrap.dedent("""
                <a href="{0}">Cluster master interface</a>
            """).format(self.cluster_master_api_url).strip()

    @property
    def config_task(self):
        return self.get_last_released_resource_task(self.__config_resource_type)

    @property
    def webscripts_task(self):
        if self.Parameters.BuildWebscriptsTask:
            return self.Parameters.BuildWebscriptsTask

        return self.get_last_released_resource_task(self.__webscripts_resource_type)

    def get_last_released_resource_task(self, resource_type):
        resource = get_last_resource_with_attrs(resource_type, all_attrs=True, attrs={'released': 'stable'})
        task_id = resource.task_id
        # sdk2/sdk1 resources mess up
        sdk2_resource = sdk2.Resource.find(task_id=task_id, resource_type=resource_type).first()
        return sdk2_resource.task

    def get_tasks_resources(self, tasks):
        resources = []
        for task in tasks:
            input_resources = sdk2.Resource.find(task_id=task.id, type=task.get_resource_type()).limit(1000)

            task_resources = [
                {
                    'id': r.id,
                    'type': r.type.name
                } for r in input_resources
            ]

            resources.extend(task_resources)

        logging.info('list of resources to upload: {}'.format(resources))
        return resources

    def get_resources(self):
        resources = self.get_tasks_resources([
            self.Parameters.BuildBundleTask,
            self.config_task,
            self.webscripts_task
        ])
        return resources

    def get_variables(self):
        return {"YT_PREFIX": self.yt_prefix}

    def create_new_yt_prefix(self):
        import yt.wrapper as yt

        yt_client = yt.YtClient(proxy="arnold.yt.yandex.net", token=sdk2.Vault.data('VIDEODEV', 'yt_token'))

        logging.info('Goint to copy testing directory to "{0}"'.format(self.yt_prefix[:-1]))
        yt_client.copy(self.__STABLE_TESTING_DIRECTORY, self.yt_prefix[:-1])  # remove last character '/' from new yt_prefix

    def update_deploy_unit(self):
        from sandbox.projects.VideoSearch.deploy_helper import DeployUnitDeployer

        variables_to_update = self.get_variables()
        resources_to_update = self.get_resources()

        deployer = DeployUnitDeployer(
            'video-robot-cm-video-docbase-test',
            'video-docbase-template',
            'video-docbase-{0}'.format(self.timestamp)
        )
        for variable_name, variable_value in variables_to_update.iteritems():
            deployer.update_literal_variable('video-docbase', variable_name, variable_value)

        for resource in resources_to_update:
            if resource['type'] in self.__resource_type_to_deploy_resource:
                deployer.update_static_resource(
                    self.__resource_type_to_deploy_resource[resource['type']],
                    'sbr:{0}'.format(resource['id'])
                )

        deployer.deploy()
        deployer.wait_for_deploy()

    def update_awacs(self):
        from sandbox.projects.VideoSearch.awacs_helper import AwacsUpdater

        awacs_updater = AwacsUpdater(
            'video-robot-cm-test.yandex.net',
            'video-robot-cm-test.yandex.net_vla'
        )
        awacs_updater.create_backend(
            'video-robot-cm-video-docbase-test-{0}-backend'.format(self.timestamp),
            'video-robot-cm-video-docbase-test-template-backend',
            'video-robot-cm-video-docbase-test.video-docbase-{0}'.format(self.timestamp)
        )
        awacs_updater.create_upstream(
            'docbase-test-{0}-video-robot-cm-test_in_yandex-team_ru'.format(self.timestamp),
            'docbase-test-template-video-robot-cm-test_in_yandex-team_ru',
            '/docbase/{0}(/.*)?'.format(self.timestamp),
            'video-robot-cm-video-docbase-test-{0}-backend'.format(self.timestamp)
        )
        awacs_updater.wait_backend('video-robot-cm-video-docbase-test-{0}-backend'.format(self.timestamp))

    def on_execute(self):
        if not self.Context.yt_prefix:
            self.Context.yt_prefix = self.yt_prefix
        if not self.Context.BuildDocbaseBundleTask:
            self.Context.BuildDocbaseBundleTask = self.Parameters.BuildBundleTask.id
        if not self.Context.BuildWebscriptsTask:
            self.Context.BuildWebscriptsTask = self.Parameters.BuildWebscriptsTask.id

        cluster_master_api_client = ClusterMasterApi('{0}{1}'.format(self.__CLUSTER_MASTER_API_URL, self.timestamp))

        with self.memoize_stage.first_step:

            self.update_deploy_unit()
            self.update_awacs()
            self.create_new_yt_prefix()

            cluster_master_api_client.invalidate_whole_path('Start')
            cluster_master_api_client.run_whole_path('Swap-States')
            time.sleep(60)  # wait for statuses to update

        status = cluster_master_api_client.check_target_status('Swap-States')
        if status == ClusterMasterResponceStatus.WAITING:
            logging.info("Docbase building still in action. Wait for 5 minutes")
            raise sdk2.WaitTime(600)

        if status == ClusterMasterResponceStatus.FAILURE:
            logging.info("Docbase building failed.")
            raise common.errors.TaskFailure("Docbase building failed")

    def on_save(self):
        self.Requirements.tasks_resource = SandboxTasksBinary.find(attrs={
            'type': 'VIDEO_CREATE_TEST_ENV'
        }).first().id
