import time
import json

import util


def produce_timestamped_promo_trie(proxy, path_dict, trie_file_name, output_file_name):
    timestamped_promo_dict = _load_timestamped_promo_dict(proxy, path_dict)
    _process_trie(trie_file_name, output_file_name, timestamped_promo_dict)


def _load_timestamped_promo_dict(proxy, path_dict):
    path_to_timestamped_promo = {}
    for key in path_dict:
        path = path_dict[key]
        if path not in path_to_timestamped_promo:
            path_to_timestamped_promo[path] = _load_timestamped_promo(proxy, path_dict[key])

    timestamped_promo_dict = {}
    for key in path_dict:
        path = path_dict[key]
        timestamped_promo_dict[key] = path_to_timestamped_promo[path]

    return timestamped_promo_dict


def _load_timestamped_promo(proxy, path):
    import yt.wrapper as yt

    yt.config['proxy']['url'] = proxy
    if not yt.exists(path):
        return []

    timestamped_promo = []
    for row in yt.read_table(path):
        promo = {
            "url": row["url"],
            "tm_start": row["tm_start"],
            "tm_end": row["tm_end"]
        }
        timestamped_promo.append(promo)

    return timestamped_promo


def _process_trie(trie_file_name, output_file_name, timestamped_promo_dict):
    with open(trie_file_name, "r") as trie_file:
        trie = trie_file.readlines()

        new_trie = []
        timestamp = time.time()
        for line in trie:
            new_trie.append(_process_trie_line(line, timestamped_promo_dict, timestamp))

    with open(output_file_name, "w") as trie_file:
        trie_file.write("\n".join(new_trie))


def _process_trie_line(line, timestamped_promo_dict, timestamp):
    line = line.strip()
    try:
        key, value = util.parse_trie_line(line)
    except IndexError:
        return ""

    if key not in timestamped_promo_dict:
        return line

    urls_to_add = []
    for promo in timestamped_promo_dict[key]:
        if timestamp <= promo["tm_start"] or timestamp > promo["tm_end"]:
            continue
        urls_to_add.append(promo)

    if not urls_to_add:
        return line

    value_dict = json.loads(value)
    if "mixed_promo_url_data" not in value_dict:
        value_dict["mixed_promo_url_data"] = {}
    if "docs" not in value_dict["mixed_promo_url_data"]:
        value_dict["mixed_promo_url_data"]["docs"] = []
    value_dict["mixed_promo_url_data"]["docs"].extend(urls_to_add)

    line_arr = [key, "*", "0", json.dumps(value_dict)]

    return "\t".join(line_arr)


if __name__ == "__main__":
    produce_timestamped_promo_trie(
        "hahn",
        {"videohub;prod": "//home/videoindex/recommender/custom_tags/yandex_show_timestamped_promo"},
        "videohub.fast",
        "new_trie"
    )
