from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox import sdk2
import logging
import sandbox.common.types.task as ctt
import time

FPS_SERVER_CONFIG = \
"""EnableAppHost: true
AppHostThreads: 32
Port: 80
BindAddress: "*"
"""


class FPSConfig(sdk2.Resource):
    """ Config for --conf """


class FPSServerConfig(sdk2.Resource):
    """ Config for --server_conf """


class FPSRulesConfig(sdk2.Resource):
    """ Config for --rules_conf """


class FPSDefaultQueries(sdk2.Resource):
    """ Default queries file  """


class FPSExternalDelayedViewQueries(sdk2.Resource):
    """ Queries with recommender_experiment=external_delayed_view """


class FPSOfflineRecommenderQueries(sdk2.Resource):
    """ Queries with recommender_experiment=offline_recommender """


class VideoTestFreshProfileServer(binary_task.LastBinaryTaskRelease, sdk2.Task):
    location_name = "SAS"
    default_custom_files = {
            "784564381": "vhs_history_dssm_outer.dssm",
            "785619407": "vhs_history_dssm_outer.dssm_params",
            "913534615": "entity_recommender_history_1.dssm",
            "913534896": "entity_recommender_history_1.dssm_params",
            "1006605639": "ether_history_outer.dssm",
            "1006610143": "ether_history_outer.dssm_params",
            "1072051010": "web-dt-300-15.ckmeans",
            "1072052530": "video-dc-qt-120-15.ckmeans",
            "1216531196": "ether_history_v1_outer.dssm",
            "1216533432": "ether_history_v1_outer.dssm_params",
            "1408746556": "ether_history_v2_outer.dssm",
            "1408752050": "ether_history_v2_outer.dssm_params"
    }

    class Parameters(sdk2.Task.Parameters):
        fps_resource = sdk2.parameters.Resource('Fresh profile server binary', required=True)

        config_resource = sdk2.parameters.Resource('Fresh profile server config', required=True)

        custom_files = sdk2.parameters.Dict('Additional resources (id - path)')

        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        params = self.Parameters
        config = FPSConfig(self, "FPS config", "processor.conf")
        config_data = sdk2.ResourceData(config)
        self._patch_config(sdk2.ResourceData(params.config_resource), config_data)

        rules_resource = FPSRulesConfig(self, "FPS rules config", "rules.conf")
        rules_resource_data = sdk2.ResourceData(rules_resource)
        rules_resource_data.path.write_bytes("")
        rules_resource_data.ready()

        server_config_resource = FPSServerConfig(self, "FPS server config", "server.conf")
        server_config_resource_data = sdk2.ResourceData(server_config_resource)
        server_config_resource_data.path.write_bytes(FPS_SERVER_CONFIG)
        server_config_resource_data.ready()

        self.test_queries = self._get_test_queries()

        query_tasks, task_name, task_status = self._create_test_tasks(
            fps_resource=params.fps_resource,
            config_resource=config,
            rules_resource=rules_resource,
            server_config_resource=server_config_resource,
            custom_files=self._patch_custom_files(params.custom_files),
            owner=params.owner
        )
        tasks_to_wait = len(query_tasks)

        while tasks_to_wait:
            for i in range(len(query_tasks)):
                cur_status = self._get_task_status(query_tasks[i], task_name[i], task_status[i])
                if task_status[i] == cur_status:
                    continue
                task_status[i] = cur_status
                if cur_status == "SUCCESS":
                    tasks_to_wait -= 1
                    continue
                query_tasks[i] = self._create_accept_task(query_tasks[i].id, task_name[i], params.owner)
            time.sleep(5)

    def _patch_config(self, from_config, to_config):
        result_config = ""
        with open(str(from_config.path), "r") as conf_file:
            for line in conf_file:
                result_config += line.replace("#LOCATION#", self.location_name.lower())
        to_config.path.write_bytes(result_config)
        to_config.ready()
        logging.info("Patched server config")

    def _create_test_tasks(self, fps_resource, config_resource, rules_resource,
                            server_config_resource, custom_files, owner):
        from sandbox.projects.dj.DjTestFreshProfileServer import DjTestFreshProfileServer

        tesk_tasks = []

        for exp_name in self.test_queries:
            logging.info("Preparing test {}".format(exp_name))
            test_task = DjTestFreshProfileServer(
                self,
                fps_resource=fps_resource,
                config_resource=config_resource,
                rules_resource=rules_resource,
                server_config_resource=server_config_resource,
                custom_files=custom_files,
                shard_dir="resources",
                queries_resource=self.test_queries[exp_name]["resource_id"],
                __requirements__={"client_tags": "GENERIC & {}".format(self.location_name), "ram": 3 * 1024},
                description="Child of {}".format(self.id),
                owner=owner
            ).enqueue()
            tesk_tasks.append(test_task)

        return tesk_tasks, [task_name for task_name in self.test_queries], ["TEST", "TEST", "TEST"]

    def _create_accept_task(self, test_task_id, task_name, owner):
        from sandbox.projects.dj.DjAcceptFreshProfileServer import DjAcceptFreshProfileServer

        logging.info("Creating ACCEPT for {}".format(task_name))

        return DjAcceptFreshProfileServer(
            self,
            test_task=test_task_id,
            latency_avg_limit=self.test_queries[task_name]["latency_avg_limit"],
            latency_p99_limit=self.test_queries[task_name]["latency_p99_limit"],
            max_rss_limit=self.test_queries[task_name]["max_rss_limit"],
            failed_requests_limit=self.test_queries[task_name]["failed_requests_limit"],
            __requirements__={"client_tags": "GENERIC & {}".format(self.location_name)},
            description="Child of {}".format(self.id),
            owner=owner
        ).enqueue()

    def _get_task_status(self, task, task_name, cur_status):
        task.reload()
        if task.status in ctt.Status.Group.SCHEDULER_FAILURE | ctt.Status.Group.FAIL_ON_ANY_ERROR:
            raise TaskFailure("{} failed at stage {}".format(task_name, cur_status))
        elif task.status in ctt.Status.Group.BREAK | ctt.Status.Group.FINISH:
            return "ACCEPT" if cur_status == "TEST" else "SUCCESS"
        return cur_status

    def _get_test_queries(self):
        return {
            "default": {
                "resource_id": sdk2.Resource.find(type="FPSDEFAULT_QUERIES").order(-sdk2.Resource.id).first().id,
                "latency_avg_limit": 25,
                "latency_p99_limit": 62,
                "max_rss_limit": 15000,
                "failed_requests_limit": 0.005
            },
            "offline_recommender": {
                "resource_id": sdk2.Resource.find(type="FPSEXTERNAL_DELAYED_VIEW_QUERIES").order(-sdk2.Resource.id).first().id,
                "latency_avg_limit": 15,
                "latency_p99_limit": 50,
                "max_rss_limit": 15000,
                "failed_requests_limit": 0.005
            },
            "external_delayed_view": {
                "resource_id": sdk2.Resource.find(type="FPSOFFLINE_RECOMMENDER_QUERIES").order(-sdk2.Resource.id).first().id,
                "latency_avg_limit": 15,
                "latency_p99_limit": 50,
                "max_rss_limit": 15000,
                "failed_requests_limit": 0.005
            }
        }

    def _patch_custom_files(self, param_custom_files):
        self.default_custom_files[sdk2.Resource.find(released="stable", type="VIDEO_FRESH_PROFILE_SERVER_METADATA_SHARD", state="READY").order(-sdk2.Resource.id).first().id] = "resources/episode_shard/metadata_shard.bin"
        file_names = set()
        custom_files = {}
        for resource_id in param_custom_files:
            custom_files[resource_id] = param_custom_files[resource_id]
            file_names.add(custom_files[resource_id])
        for resource_id in self.default_custom_files:
            if self.default_custom_files[resource_id] not in file_names:
                custom_files[resource_id] = self.default_custom_files[resource_id]
        return custom_files

