# -*- coding: utf-8 -*-

import os
import shutil
import logging
from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.VideoSearch import video_resource_types


VIDEOHUB_CONFIG_FILES = [
    ("", "{}")
]

ENTITY_CONFIG_FILES = [
    ("user/entity/update_profile_action_rules.conf", "{}/update_profile_action_rules.conf"),
    ("user/entity/update_counter_from_profile_rules.conf", "{}/update_counter_from_profile_rules.conf"),
    ("user/entity/update_embedding_from_profile_rules.conf", "{}/update_embedding_from_profile_rules.conf"),
    ("user/entity/update_profile_action_rules_latest_ratings.conf", "{}/update_profile_action_rules_latest_ratings.conf"),
]

JUNK_TO_REMOVE = ["user/README", "user/BUGS"]


class VideoVideohubCreateDjBundle(nanny.ReleaseToNannyTask2, sdk2.Task):

    task = 'VIDEO_VIDEOHUB_CREATE_DJ_BUNDLE'

    CHECKOUT_PATH = "bundle"

    class Parameters(sdk2.Task.Parameters):
        videohub_arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Videohub arcadia URL; must always point to Arcadia root (branches/tags are allowed). If empty, trunk URL is used."
        )
        videohub_config_path = sdk2.parameters.String(
            "Videohub Config Path",
            required=True,
            default="extsearch/video/quality/recommender/dj/rules"
        )
        entity_arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Entity arcadia URL; must always point to Arcadia root (branches/tags are allowed). If empty, trunk URL is used."
        )
        entity_config_path = sdk2.parameters.String(
            "Entity User Config Path",
            required=True,
            default="entity/recommender/config"
        )

    def on_execute(self):
        tmp_path = os.path.realpath(self.CHECKOUT_PATH)
        logging.info("TMP_PATH = {}".format(tmp_path))

        if not os.path.exists(tmp_path):
            os.mkdir(tmp_path)
            logging.info("TMP_PATH CREATED")

        self._export_configs(
            VIDEOHUB_CONFIG_FILES,
            self.Parameters.videohub_arcadia_url,
            self.Parameters.videohub_config_path,
            tmp_path
        )
        self._export_configs(
            ENTITY_CONFIG_FILES,
            self.Parameters.entity_arcadia_url,
            self.Parameters.entity_config_path,
            tmp_path
        )
        self._remove_junk(tmp_path)

        path_to_tar = self._tar_dir(tmp_path)
        resource = video_resource_types.VIDEO_VIDEOHUB_DJ_CONFIG_BUNDLE(
            self,
            "config bundle tar",
            "dj-config_bundle.tar.gz"
        )
        data = sdk2.ResourceData(resource)

        with open(path_to_tar, "rb") as f:
            data.path.write_bytes(f.read())

        data.ready()

    def _remove_junk(self, path):
        for name in JUNK_TO_REMOVE:
            os.remove(os.path.join(path, name))

    def _join_arcadia_paths(self, base_path, target_path):
        joined_path = os.path.join(
            sdk2.svn.Arcadia.parse_url(base_path).path,
            target_path
        )
        return sdk2.svn.Arcadia.replace(base_path, joined_path)

    def _export_configs(self, templates, arcadia_url, config_path, path):
        for name, template in templates:
            arcadia_path = self._join_arcadia_paths(arcadia_url, template.format(config_path))
            logging.info("PATH = {}".format(os.path.join(path, name)))
            sdk2.svn.Arcadia.export(arcadia_path, os.path.join(path, name))

    def _tar_dir(self, path):
        return shutil.make_archive("dj-config_bundle", "gztar", path)
