import time
import nanny_rpc_client
from nanny_rpc_client.exceptions import NotFoundError
from infra.awacs.proto import api_pb2, api_stub
from sandbox import sdk2


AWACS_RPC_URL = 'https://awacs.yandex-team.ru/api/'

class AwacsUpdater:

    def __init__(self, namespace_id, balancer_id):
        self.__namespace_id = namespace_id
        self.__balancer_id = balancer_id

    __OAUTH_VAULT_OWNER = 'VIDEO-TESTENV'

    @property
    def oauth_token(self):
        return sdk2.Vault.data(self.__OAUTH_VAULT_OWNER, "AWACS_TOKEN")

    @property
    def backend_service(self):
        awacs_rpc = nanny_rpc_client.RequestsRpcClient(rpc_url=AWACS_RPC_URL, oauth_token=self.oauth_token)
        return api_stub.BackendServiceStub(awacs_rpc)

    @property
    def upstream_service(self):
        awacs_rpc = nanny_rpc_client.RequestsRpcClient(rpc_url=AWACS_RPC_URL, oauth_token=self.oauth_token)
        return api_stub.UpstreamServiceStub(awacs_rpc)

    @property
    def domain_service(self):
        awacs_rpc = nanny_rpc_client.RequestsRpcClient(rpc_url=AWACS_RPC_URL, oauth_token=self.oauth_token)
        return api_stub.DomainServiceStub(awacs_rpc)

    def get_backend(self, backend_id):
        req = api_pb2.GetBackendRequest()
        req.namespace_id = self.__namespace_id
        req.id = backend_id

        return self.backend_service.get_backend(req).backend

    def try_get_backend(self, backend_id):
        try:
            return self.get_backend(backend_id)
        except NotFoundError:
            return None

    def create_backend(self, backend_id, template_backend_id, yp_endpoint_set_id):
        template_backend = self.get_backend(template_backend_id)

        req = api_pb2.CreateBackendRequest()
        req.meta.CopyFrom(template_backend.meta)
        req.meta.id = backend_id
        req.spec.CopyFrom(template_backend.spec)
        req.spec.selector.yp_endpoint_sets[0].endpoint_set_id = yp_endpoint_set_id

        return self.backend_service.create_backend(req)

    def wait_backend(self, backend_id):
        ready = False
        while not ready:
            time.sleep(60)
            backend = self.get_backend(backend_id)
            ready = (backend.statuses[0].active['{0}:{1}'.format(self.__namespace_id, self.__balancer_id)].status == 'True')

    def remove_backend(self, backend_id):
        backend = self.try_get_backend(backend_id)
        if backend is None:
            return
        req  = api_pb2.RemoveBackendRequest()
        req.id = backend.meta.id
        req.namespace_id = backend.meta.namespace_id
        req.version = backend.meta.version

        return self.backend_service.remove_backend(req)

    def wait_for_remove_backend(self, backend_id):
        while self.try_get_backend(backend_id) is not None:
            time.sleep(60)

    def get_upstream(self, upstream_id):
        req = api_pb2.GetUpstreamRequest()
        req.namespace_id = self.__namespace_id
        req.id = upstream_id

        return self.upstream_service.get_upstream(req).upstream

    def try_get_upstream(self, upstream_id):
        try:
            return self.get_upstream(upstream_id)
        except NotFoundError:
            return None

    def create_upstream(self, upstream_id, template_upstream_id, matcher_path, backend_id):
        template_upstream = self.get_upstream(template_upstream_id)

        req = api_pb2.CreateUpstreamRequest()
        req.meta.CopyFrom(template_upstream.meta)
        req.meta.id = upstream_id
        req.spec.CopyFrom(template_upstream.spec)
        req.spec.yandex_balancer.config.l7_upstream_macro.id = upstream_id

        req.spec.yandex_balancer.config.l7_upstream_macro.matcher.path_re = matcher_path
        req.spec.yandex_balancer.config.l7_upstream_macro.monitoring.uuid = upstream_id
        req.spec.yandex_balancer.config.l7_upstream_macro.rewrite[0].pattern.re = matcher_path
        req.spec.yandex_balancer.config.l7_upstream_macro.by_dc_scheme.dcs[0].backend_ids[0] = backend_id
        req.spec.yandex_balancer.yaml = ''

        return self.upstream_service.create_upstream(req)

    def get_domain(self, domain_id):
        req = api_pb2.GetDomainRequest()
        req.namespace_id = self.__namespace_id
        req.id = domain_id

        return self.domain_service.get_domain(req).domain

    def remove_upstream(self, upstream_id):
        upstream = self.try_get_upstream(upstream_id)
        if upstream is None:
            return
        req = api_pb2.RemoveUpstreamRequest()
        req.id = upstream.meta.id
        req.namespace_id = upstream.meta.namespace_id
        req.version = upstream.meta.version

        return self.upstream_service.remove_upstream(req)

    def wait_for_remove_upstream(self, upstream_id):
        while self.try_get_upstream(upstream_id) is not None:
            time.sleep(60)
