import time
import yp.data_model as data_model
from infra.dctl.src.lib import yp_client
from infra.dctl.src.lib import stage
from infra.dctl.src import consts
import library.python.vault_client as vault_client
from sandbox import sdk2


YP_UPDATE_CLUSTER = 'xdc'

def clear_read_only_fields(obj):
    obj.ClearField('status')
    obj.meta.ClearField('uuid')
    obj.meta.ClearField('type')
    obj.meta.ClearField('fqid')
    obj.meta.ClearField('creation_time')
    obj.ClearField('annotations')

def get_yp_client(yp_cluster, yp_token):
    cluster = yp_cluster.lower()
    yp_address = consts.CLUSTER_CONFIGS[cluster].address
    client = yp_client.YpClient(url=yp_address, token=yp_token)
    return client


class DeployUnitWorker(object):

    def __init__(self, deploy_stage):
        self._deploy_stage_id = deploy_stage
        self.__load_stage()

    _OAUTH_VAULT_OWNER = 'VIDEO-TESTENV'
    _YP_UPDATE_CLUSTER = 'xdc'

    @property
    def oauth_token(self):
        return sdk2.Vault.data(self._OAUTH_VAULT_OWNER, "DEPLOY_TOKEN")

    def __get_vault_client_rsa_fallback(self):
        client_rsa_fallback = vault_client.VaultClient(
            host=consts.VAULT_HOST,
            check_status=False,
            rsa_auth=True,
            rsa_login='robot-video-acc'
        )
        return client_rsa_fallback

    def __get_vault_client(self):
        client = vault_client.VaultClient(
            host=consts.VAULT_HOST,
            check_status=False,
            authorization=self.oauth_token
        )
        return client

    def __load_stage(self):
        client = get_yp_client(self._YP_UPDATE_CLUSTER, self.oauth_token)
        stage_cfg = stage.get(self._deploy_stage_id, client)
        clear_read_only_fields(stage_cfg)
        self._stage_cfg = stage_cfg

    def _run(self):
        client = get_yp_client(self._YP_UPDATE_CLUSTER, self.oauth_token)
        stage.put(
            stage=self._stage_cfg,
            cluster=self._YP_UPDATE_CLUSTER,
            rewrite_delegation_tokens=True,
            vault_client=self.__get_vault_client(),
            vault_client_rsa_fallback=self.__get_vault_client_rsa_fallback(),
            client=client
        )


class DeployUnitDeployer(DeployUnitWorker):

    def __init__(self, deploy_stage, template_deploy_unit, deploy_unit_id):
        super(DeployUnitDeployer, self).__init__(deploy_stage)
        self.__deploy_unit_id = deploy_unit_id
        self.__template_deploy_unit = template_deploy_unit
        self.__clone_template_deploy_unit()

    def __clone_template_deploy_unit(self):
        self._stage_cfg.spec.deploy_units[self.__deploy_unit_id].CopyFrom(self._stage_cfg.spec.deploy_units[self.__template_deploy_unit])
        self.__deploy_unit = self._stage_cfg.spec.deploy_units[self.__deploy_unit_id]
        self.__deploy_unit.revision = 1

    def __get_pod_spec(self):
        return self.__deploy_unit.multi_cluster_replica_set.replica_set.pod_template_spec.spec.pod_agent_payload.spec

    def __get_deploy_unit_static_resources(self):
        return self.__get_pod_spec().resources.static_resources

    def __get_box_env(self, box_id):
        boxes = self.__get_pod_spec().boxes
        for box in boxes:
            if box.id == box_id:
                return box.env
        return None

    def update_static_resource(self, resource_id, sandbox_resource_id, sandbox_resource_checksum='EMPTY:'):
        static_resources = self.__get_deploy_unit_static_resources()
        for resource in static_resources:
            if resource.id == resource_id:
                resource.url = sandbox_resource_id
                resource.verification.checksum = sandbox_resource_checksum

    def update_literal_variable(self, box_id, variable_name, variable_value):
        box_env = self.__get_box_env(box_id)
        for variable in box_env:
            if variable.name == variable_name:
                variable.value.literal_env.value = variable_value

    def deploy(self):
        self._run()

    def wait_for_deploy(self):
        client = get_yp_client(self._YP_UPDATE_CLUSTER, self.oauth_token)
        ready = False
        while not ready:
            time.sleep(60)

            stage_cfg = stage.get(self._deploy_stage_id, client)
            state = stage_cfg.status.deploy_units[self.__deploy_unit_id]
            ready = (state.ready.status == data_model.CS_TRUE)


class DeployUnitRemover(DeployUnitWorker):

    def __init__(self, deploy_stage, deploy_unit_id):
        super(DeployUnitRemover, self).__init__(deploy_stage)
        self.__deploy_unit_id = deploy_unit_id
        self.__remove_deploy_unit()

    def __remove_deploy_unit(self):
        if self.__deploy_unit_id in self._stage_cfg.spec.deploy_units:
            del self._stage_cfg.spec.deploy_units[self.__deploy_unit_id]

    def remove(self):
        self._run()

    def wait_for_remove(self):
        client = get_yp_client(self._YP_UPDATE_CLUSTER, self.oauth_token)
        stage_cfg = stage.get(self._deploy_stage_id, client)
        while self.__deploy_unit_id in stage_cfg.status.deploy_units:
            time.sleep(60)

            stage_cfg = stage.get(self._deploy_stage_id, client)
