from sandbox.projects.common.platform_api.json_api_client import JsonApiClient
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2
import logging

import json
import time
import os
import requests
import copy


class PlatformDeployer(object):

    def __init__(self, env_id, component_name, template_component_name, variables_to_deploy={}, resources_to_deploy={}, new_component=True):
        self.__env_id = env_id
        self.__variables_to_deploy = variables_to_deploy
        self.__resources_to_deploy = resources_to_deploy
        self.__new_component_name = component_name
        self.__template_component_name = template_component_name
        self.__environment = self.api_client("environment/dump")(self.__env_id).get()
        self.__comments = []

        if new_component:
            self.__create_new_component()
        else:
            assert component_name == template_component_name

        logging.info(json.dumps(self.__environment, indent=4, sort_keys=True))

    __PLATFORM_URL = 'https://platform.yandex-team.ru/api/v1'
    __OAUTH_VAULT_OWNER = 'VIDEO-TESTENV'

    @property
    def oauth_token(self):
        return sdk2.Vault.data(self.__OAUTH_VAULT_OWNER, "OAUTH_TOKEN")

    @property
    def api_client(self):
        return JsonApiClient(url=self.__PLATFORM_URL,
            oauth_token=self.oauth_token)

    def __create_new_component(self):
        template_component = self.__environment['components'][0]
        for component in self.__environment['components']:
            if component['componentName'] == self.__template_component_name:
                template_component = copy.deepcopy(component)

        template_component['componentName'] = self.__new_component_name
        self.__environment['components'].append(template_component)

        template_route_settings = self.__environment['routeSettings'][0]
        for route_settings in self.__environment['routeSettings']:
            if route_settings['componentName'] == self.__template_component_name:
                template_route_settings = copy.deepcopy(route_settings)

        template_route_settings['location'] = '/{}'.format(self.__new_component_name)
        template_route_settings['componentName'] = self.__new_component_name
        self.__environment['routeSettings'].append(template_route_settings)
        self.__comments.append('add {}'.format(self.__new_component_name))

    def __update_environment_resources(self):
        new_resources = self.__resources_to_deploy
        self.__comments.append(", ".join(map(lambda r: "{type}={id}".format(type=r['type'], id=r['id']), new_resources)))
        for component in self.__environment['components']:
            if component['componentName'] == self.__new_component_name:
                for existing_resource in component['sandboxResources']:
                    res_info = sdk2.Resource.find(id=existing_resource['id']).first()

                    if res_info is None:
                        continue

                    for new_resource in new_resources:
                        if res_info.type.name == new_resource['type']:
                            existing_resource['id'] = new_resource['id']

    def __update_environment_variables(self):
        new_variables = self.__variables_to_deploy
        logging.info(new_variables)
        for component in self.__environment['components']:
            if component['componentName'] == self.__new_component_name:
                for variable_name, variable_value in new_variables.iteritems():
                    component['environmentVariables'][variable_name] = variable_value

    def deploy(self):
        self.__update_environment_resources()
        self.__update_environment_variables()
        logging.info('Environment to deploy:')
        self.__environment['comment'] = "; ".join(self.__comments)
        logging.info(json.dumps(self.__environment, indent=4, sort_keys=True))
        self.__deploy_result = self.api_client("environment/upload").post(self.__environment)
        logging.info('Deploy result: {} '.format(json.dumps(self.__deploy_result, indent=4, sort_keys=True)))

    def wait_for_deploy(self):
        deployed = False
        sleep_time = 0
        while not deployed:
            logging.info('Wait for deploy for a {} seconds'.format(sleep_time))
            time.sleep(sleep_time)
            sleep_time = min(sleep_time * 2 + 1, 180)

            answer = self.api_client("environment/status")(self.__env_id).get()
            logging.info('Result: {}'.format(json.dumps(answer, indent=4, sort_keys=True)))

            deployed = answer['status'] == 'DEPLOYED'


class ClusterMasterResponceStatus:
    SUCCESS = 1
    FAILURE = 2
    WAITING = 3


class ClusterMasterApi(object):

    def __init__(self, cluster_master_url):
        self.__cluster_master_url = cluster_master_url
        self._initialize_session()

    def _initialize_session(self):
        self._session = requests.Session()
        CA_CERTS = '/etc/ssl/certs/ca-certificates.crt'
        if os.path.isfile(CA_CERTS):
            self._session.verify = CA_CERTS

        retries = Retry(total=3, backoff_factor=1.0, status_forcelist=[429, 500, 502, 503, 504])
        self._session.mount(self.__cluster_master_url, HTTPAdapter(max_retries=retries))
        self._session.headers.update({
            'Content-Type': 'application/json',
            })

    def __perform_action(self, action, target):
        handler = '{0}/command/{1}?target={2}'.format(self.__cluster_master_url, action, target)
        logging.info('Calling url: {}'.format(handler))
        self._session.get(handler)

    def run_whole_path(self, target):
        self.__perform_action('run-path', target)

    def invalidate_whole_path(self, target):
        self.__perform_action('invalidate-path', target)

    def set_variable(self, variable_name, variable_value):
        self._session.get('{0}/variables/set?name={1}&value={2}'.format(self.__cluster_master_url, variable_name, variable_value))

    def check_target_status(self, target):
        handler = '{0}/targetstatus/{1}'.format(self.__cluster_master_url, target)
        target_status = self._session.get(handler).content
        logging.info('Target {0} now in the status {1}'.format(target, target_status))
        logging.info('Called url: {}'.format(handler))
        wait_statuses = ['pending', 'running']

        if target_status in wait_statuses:
            return ClusterMasterResponceStatus.WAITING

        if target_status != 'success':
            return ClusterMasterResponceStatus.FAILURE

        return ClusterMasterResponceStatus.SUCCESS

    def disable_retry_on_success(self, target):
        self._session.get('{}//target_toggle_restart/{}/restart_on_success?action=disable'.format(self.__cluster_master_url, target))
