# -*- coding: utf-8 -*-

import re
import requests

from kazoo.client import KazooClient

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common.nanny import nanny

from sandbox.projects.DeployNannyShardmap import NannyServiceNameParameter

from sandbox.sandboxsdk.sandboxapi import Sandbox
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class ZKClient(object):
    ZK_HOSTS = ('zookeeper1.search.yandex.net:2181,'
                'zookeeper2.search.yandex.net:2181,'
                'zookeeper3.search.yandex.net:2181,'
                'zookeeper4.search.yandex.net:2181,'
                'zookeeper5.search.yandex.net:2181')

    def __enter__(self):
        self.kz = KazooClient(hosts=self.ZK_HOSTS, read_only=False)
        self.kz.start()
        return self.kz

    def __exit__(self, type, value, traceback):
        self.kz.stop()


class VideoSetMediaZkFlags(SandboxTask):
    """
        Проверяем состояние сервиса в няни и выставляем
        кастомные флаги в поисковом зукипере
    """

    type = 'VIDEO_SET_MEDIA_ZK_FLAGS'

    input_parameters = [
        NannyServiceNameParameter,
    ]

    def _get_nanny_attrs_by_snapshot(self, nanny_client, snapshot_id):
        """
            Получить атрибуты рантайма по снепшоту из сервисов 2.0
            (этого нет в стандартной библиотеке nanny)
        """
        resources_url = nanny_client._api_url + \
            '/v2/history/services/runtime_attrs/' + snapshot_id
        try:
            r = nanny_client._session.get(resources_url)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise nanny._handle_http_error(e)
        except Exception as e:
            raise nanny.NannyApiException(
                'Failed to call Nanny API. Error: {}'.format(e))

        return r.json()

    def _get_shardmap_name(self, runtime_attrs):
        # get shardmap name
        service_shardmap_dump = runtime_attrs[
            u'content'][u'resources'][u'sandbox_bsc_shard']
        if 'SANDBOX_SHARDMAP' in service_shardmap_dump[u'chosen_type']:
            production_shardmap_task_id = service_shardmap_dump[
                u'sandbox_shardmap'][u'task_id']
            shardmap_resource_type_str = service_shardmap_dump[
                u'sandbox_shardmap'][u'resource_type']
            shardmap_resource_type = getattr(resource_types, shardmap_resource_type_str)
            production_resource_id = apihelpers.get_task_resource_id(
                production_shardmap_task_id,
                shardmap_resource_type)
            production_shardmap_name = Sandbox().get_resource_attribute(
                production_resource_id,
                u'shardmap_name')
            if not production_shardmap_name:
                raise SandboxTaskFailureError(
                    "Couldn't get sandbox id with production shardmap for service %s" %
                    (self.ctx[NannyServiceNameParameter.name]))
            else:
                return production_shardmap_name
        else:
            raise SandboxTaskFailureError(
                "Selected service doesn't use shardmaps")

    def _get_content(self, snapshot, runtime_attrs):
        time_start = snapshot[u'entered']
        time_end = time_start
        configuration_name = snapshot[u'conf_id']
        taskgroup_id = snapshot[u'taskgroup_id']
        shardmap_name = self._get_shardmap_name(runtime_attrs)
        pretty_shardmap_name = re.sub(r'\w+_shardmap', 'shardmap', shardmap_name)
        content = "%s %s %s %s %s" % (time_start, time_end, pretty_shardmap_name, configuration_name, taskgroup_id)
        return content.encode(encoding='UTF-8')

    def on_execute(self):
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token'),
        )

        current_state = nanny_client.get_service_current_state(
            self.ctx[NannyServiceNameParameter.name])
        service_status = current_state[u'content'][u'summary'][u'value']
        if service_status == u'ONLINE':
            # get runtime attr for production snapshot
            for snapshot in current_state[u'content'][u'active_snapshots']:
                if snapshot[u'state'] in (u'ACTIVE'):
                    runtime_attrs = self._get_nanny_attrs_by_snapshot(nanny_client, snapshot[u'snapshot_id'])
                    content = self._get_content(snapshot, runtime_attrs)
                    # set zookeeper for on_production
                    with ZKClient() as zk:
                        znode = "/media-services/video/flags/%s/on_production" % (self.ctx[NannyServiceNameParameter.name])
                        if not zk.exists(znode):
                            zk.create(znode, makepath=True)
                        zk.set(znode, content)
        elif service_status == u'UPDATING':
            production_snapshot_id = current_state[u'content'][u'rollback_snapshot'][u'snapshot_id']
            for snapshot in current_state[u'content'][u'active_snapshots']:
                # set zookeper for on_production_flag
                if snapshot[u'snapshot_id'] == production_snapshot_id:
                    runtime_attrs = self._get_nanny_attrs_by_snapshot(nanny_client, snapshot[u'snapshot_id'])
                    content = self._get_content(snapshot, runtime_attrs)
                    with ZKClient() as zk:
                        znode = "/media-services/video/flags/%s/on_production" % (self.ctx[NannyServiceNameParameter.name])
                        if not zk.exists(znode):
                            zk.create(znode, makepath=True)
                        zk.set(znode, content)
                # set zookeeper for preparing_flag
                elif snapshot[u'state'] in (
                        u'ACTIVATING',
                        u'GENERATING',
                        u'PREPARING'):
                    runtime_attrs = self._get_nanny_attrs_by_snapshot(nanny_client, snapshot[u'snapshot_id'])
                    content = self._get_content(snapshot, runtime_attrs)
                    with ZKClient() as zk:
                        znode = "/media-services/video/flags/%s/preparing_flag" % (self.ctx[NannyServiceNameParameter.name])
                        if not zk.exists(znode):
                            zk.create(znode, makepath=True)
                        zk.set(znode, content)


__Task__ = VideoSetMediaZkFlags
