# -*- coding: utf-8 -*-

import logging
import sandbox.common.types.misc as ctm
from sandbox.common.types.task import ReleaseStatus
import sandbox.sandboxsdk.paths as sdk_paths
from sandbox.sandboxsdk.parameters import (SandboxStringParameter, SandboxSelectParameter)
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import sandbox.projects.MediaLib.MediaStoreShardmap as mss
from sandbox.projects.common.nanny import nanny
from sandbox.projects import DeployNannyDashboard

logger = logging.getLogger()

_THUMB_CLOUD_SHARDMAP_TIMESTAMP = "shardmap_timestamp"
_THUMB_CLOUD_SHARDMAP_VERSION = "shardmap_version"

_RELEASE_TYPE = 'stable'
_DASHBOARD_NAME = 'thumbnails_video'
_DASHBOARD_FILTER = 'stable'
_SHARDMAP_DEPLOY_RECIPE = 'db_switch_sb_shardmap'


class ULtraThumbsParams:
    name = 'ultra'
    sb_resource = 'VIDEO_ULTRA_THUMB_SHARDMAP'
    shard_prefix = 'viduth'
    cloud_tier = 'VtubUltraTier0'
    filename_prefix = 'VIDUTH'


class BaseThumbsParams:
    name = 'base'
    sb_resource = 'VIDEO_THUMB_SHARDMAP'
    shard_prefix = 'vidsth'
    cloud_tier = 'VtubNidxTier0'
    filename_prefix = 'VIDSTH'


class ThdbCloudShardmapType(SandboxSelectParameter):
    name = 'thumb_shardmap_type'
    choices = [(BaseThumbsParams.name, BaseThumbsParams.name), (ULtraThumbsParams.name, ULtraThumbsParams.name)]
    default_value = BaseThumbsParams.name
    description = 'Thumb db shardmap type'


class ThumbShardsCount(SandboxStringParameter):
    name = 'thumb_shards_count'
    description = 'Shards count'
    required = True


class ThdbCloudStateGen(SandboxStringParameter):
    name = 'thumb_db_cloud_state_gen'
    description = 'Thumb db cloud state'
    required = True


class ThdbStateClean(SandboxStringParameter):
    name = 'thumb_db_state_clean'
    description = 'Thumb db state clean'
    required = True


class VideoStoreThumbsShardmap(SandboxTask, nanny.ReleaseToNannyTask):
    """Переключение базы тумбов Видео"""

    type = 'VIDEO_STORE_THUMBS_SHARDMAP'
    input_parameters = (ThdbCloudShardmapType, ThumbShardsCount, ThdbCloudStateGen, ThdbStateClean)
    cores = 1
    execution_space = 1024  # 1 Gb

    def get_thumbs_params(self):
        return {
            BaseThumbsParams.name: BaseThumbsParams,
            ULtraThumbsParams.name: ULtraThumbsParams
        }.get(self.ctx[ThdbCloudShardmapType.name], BaseThumbsParams)

    def generate_short_shardmap_filename(self):
        return "shardmap-{0}-{1}.map".format(
            self.ctx[ThdbCloudStateGen.name],
            self.ctx[ThdbStateClean.name])

    def generate_shardmap_filename(self):
        "Generate shardmap filename"
        short_name = self.generate_short_shardmap_filename()
        return "{0}_{1}".format(
            self.get_thumbs_params().filename_prefix,
            short_name)

    def shardmap_entry(self, prefix, shard, state, tier):
        template = "{prefix}-{shard:04d}-00000000-000000 {prefix}-{shard:04d}-{state} {tier}\n"
        if (self.get_thumbs_params().name == ULtraThumbsParams.name):
            template = "{prefix}-{shard:03d}-00000000-000000 {prefix}-{shard:03d}-{state} {tier}\n"
        return template.format(prefix=prefix, shard=shard,
                               state=state, tier=tier)

    def generate_entries(self, state, shardmap_file):
        count = int(self.ctx.get(ThumbShardsCount.name))
        params = self.get_thumbs_params()
        for shard in range(count):
            shardmap_file.write(self.shardmap_entry(params.shard_prefix,
                                                    shard,
                                                    state,
                                                    params.cloud_tier))

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        shardmap_filename = self.generate_shardmap_filename()
        attr_filename = self.generate_short_shardmap_filename()
        attributes = {
            _THUMB_CLOUD_SHARDMAP_TIMESTAMP: self.ctx[ThdbCloudStateGen.name],
            _THUMB_CLOUD_SHARDMAP_VERSION: self.ctx[ThdbStateClean.name],
            mss._TTL_ATTRIBUTE: mss._SHARDMAP_TTL,
            mss._SHARDMAP_NAME: attr_filename
        }

        resource = self.create_resource(
            self.descr,
            shardmap_filename,
            self.get_thumbs_params().sb_resource,
            arch=ctm.OSFamily.ANY,
            attributes=attributes
        )

        self.ctx[mss._CTX_RESULT_RESOURCE_ID] = resource.id

    def on_execute(self):
        "Download shardmap file to Sandbox"
        shardmap_filename = self.generate_shardmap_filename()
        logger.info("Going to create %s", shardmap_filename)

        sdk_paths.remove_path(shardmap_filename)

        with open(shardmap_filename, 'w') as shardmap_file:
            state = self.ctx[ThdbStateClean.name]
            self.generate_entries(state, shardmap_file)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)
        # TODO: fast workaround. Redo
        if self.get_thumbs_params().name == BaseThumbsParams.name:
            if additional_parameters['release_status'] == ReleaseStatus.STABLE:
                if self.ctx.get('switch_task_id') is not None:
                    raise SandboxTaskFailureError("Task's already been in STABLE")
                deploy_task_params = {
                    'deployment_task_id': self.id,
                    'deployment_release_status': _RELEASE_TYPE,
                    'deployment_nanny_dashboard_name': _DASHBOARD_NAME,
                    'deployment_nanny_dashboard_filter': _DASHBOARD_FILTER,
                    'deployment_nanny_dashboard_recipe': _SHARDMAP_DEPLOY_RECIPE,
                    'deployment_nanny_bool_wait': False,
                    'vault_owner': 'VIDEO-BASE-DEPLOY'
                }
                self.ctx['switch_task_id'] = self.create_subtask(
                    task_type=DeployNannyDashboard.DeployNannyDashboard.type,
                    description='Switch video base thumbs database',
                    input_parameters=deploy_task_params
                ).id
                # self.wait_all_tasks_completed(self.list_subtasks())


__Task__ = VideoStoreThumbsShardmap
