# -*- coding: utf-8 -*-

import logging
import urllib

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.sandboxapi import ARCH_LINUX
import sandbox.sandboxsdk.util as sdk_util

from sandbox.projects.common.dynamic_models.utils import compare_dicts
from sandbox.projects.common.dynamic_models.archiver import get_matrixnets_md5
from sandbox.projects.common.dynamic_models.basesearch import get_basesearch_models
from sandbox.projects.common.dynamic_models.basesearch import verify_basesearch_responses

from sandbox.projects import resource_types
from sandbox.projects.common.utils import check_if_tasks_are_ok
from sandbox.projects.common.search.settings import VideoSettings

from sandbox.projects.common import apihelpers


PRIEMKA_ATTRIBUTES = VideoSettings.RESOURCE_PRIEMKA_ATTRIBUTES
"""
    Атрибуты по которым мы определяем инструменты и базу
    для проведения тестирования
"""

EXTRA_PARAMS_BY_TYPE = {
    'full.common.hub': '&relev=vhub_request=1',
    'full.ru.hub': '&relev=vhub_request=1',
}


def _get_priemka_resource_id(resource_type):
    """
        Ищет ресурс для приёмочных испытаний

        Возвращает идентификатор ресурса или выбрасывает
        исключение SandboxTaskFailureError
    """

    resources = apihelpers.get_resources_with_attribute(
        resource_type,
        PRIEMKA_ATTRIBUTES[0],
        PRIEMKA_ATTRIBUTES[1],
        limit=1,
        arch=sdk_util.system_info()['arch']
    )

    logging.info(
        'get_resources_with_attribute(%s, %s, %s) returned %s' % (
            resource_type, PRIEMKA_ATTRIBUTES[0], PRIEMKA_ATTRIBUTES[1], resources
        )
    )

    if not resources:
        raise SandboxTaskFailureError('Cannot get %s with %s = %s' % (resource_type, PRIEMKA_ATTRIBUTES[0], PRIEMKA_ATTRIBUTES[1]))

    return resources[0].id


def _get_last_released_resource_id(resource_type, arch=None):
    """
        Ищет последнюю стабильную версию ресурса

        Возвращает идентификатор ресурса или выбрасывает
        исключение SandboxTaskFailureError
    """

    if arch is None:
        arch = sdk_util.system_info()['arch']

    resource = apihelpers.get_last_released_resource(resource_type, arch=arch)
    logging.info('get_last_released_resource(%s) returned %s' % (resource_type, resource))

    if not resource:
        raise SandboxTaskFailureError('Cannot find stable released resource %s' % resource_type)

    return resource.id


def _get_last_resource_id(resource_type, arch=None):
    """
        Ищет последнюю версию ресурса

        Возвращает идентификатор ресурса или выбрасывает
        исключение SandboxTaskFailureError
    """

    resource = apihelpers.get_last_resource(resource_type)
    if not resource:
        raise SandboxTaskFailureError('Cannot find resource %s' % resource_type)

    return resource.id


class ModelsArchiveParameter(ResourceSelector):
    """
        Архив с динамическими моделями
    """

    name = 'models_archive_resource_id'
    description = 'Models archive'
    resource_type = resource_types.VIDEO_DYNAMIC_MODELS_ARCHIVE
    required = True


class VideosearchParameter(ResourceSelector):
    """
        Базовый поиск
    """

    name = 'videosearch_resource_id'
    description = 'Videosearch executable'
    resource_type = resource_types.VIDEOSEARCH_EXECUTABLE
    required = True


class UseCustomVideosearchParameter(SandboxBoolParameter):
    """
        Базовый поиск
    """

    name = 'use_custom_videosearch'
    description = 'Use custom videosearch binary'
    default_value = False
    sub_fields = {'true': [VideosearchParameter.name]}


class VideoTestDynamicModelsArchive(SandboxTask):
    """
        Проверяет что базовый поиск корректно работает с dynamic models archive (архив с формулами):
        Находит archiver, достает из архива имена формул
        Находит базовый, достает из него имена формул
        Формирует файл запросов и запускает дочерний таск GET_BASESEARCH_RESPONSES
    """

    type = 'VIDEO_TEST_DYNAMIC_MODELS_ARCHIVE'

    archive_parameter = ModelsArchiveParameter

    def on_execute(self):
        archive_id = self.ctx[self.archive_parameter.name]
        if self.ctx[UseCustomVideosearchParameter.name]:
            videosearch_id = self.ctx[VideosearchParameter.name]
        else:
            videosearch_id = _get_last_released_resource_id(resource_types.VIDEOSEARCH_EXECUTABLE, arch=ARCH_LINUX)

        if 'test_task_id' not in self.ctx:
            # build test queries
            models = self._get_models(videosearch_id, archive_id)
            model_ids = list(m['matrixnet_id'] for m in models)

            queries_id = self._create_plain_text_queries(models, self.descr)
            self.ctx['model_ids'] = model_ids

            # run subtask
            self.ctx['test_task_id'] = self._get_basesearch_responses(videosearch_id, archive_id, queries_id)

        test_task_id = self.ctx['test_task_id']
        test_task = channel.sandbox.get_task(test_task_id)
        if not test_task.is_done():
            self.wait_task_completed(test_task_id)

        check_if_tasks_are_ok([test_task_id])
        responses_file = self.sync_resource(test_task.ctx['out_resource_id'])
        model_ids = self.ctx['model_ids']

        self.set_info('Check responses from videosearch.')
        verify_basesearch_responses(responses_file, model_ids)
        self.set_info('%s formulas checked' % len(model_ids))

    def _get_request_template(self, model):
        model_type = model['model_name'].split(':')[0]

        template = '?text=test&ms=proto'

        if model_type in EXTRA_PARAMS_BY_TYPE:
            template += EXTRA_PARAMS_BY_TYPE[model_type]

        return template

    def _create_plain_text_queries(self, models, description, output_filename='queries.txt'):
        """
            Создаёт ресурс PLAIN_TEXT_QUERIES
            на основе идентификаторов моделей
            :param models: итерируемый объект с описаниями моделей
            :param description: описание создаваемого sandbox-ресурса
            :param output_filename: имя файла создаваемого sandbox-ресурса
        """

        r = channel.task.create_resource(
            description,
            output_filename,
            resource_types.PLAIN_TEXT_QUERIES
        )

        with open(r.path, 'w') as output_file:
            for model in models:
                param = urllib.quote_plus("fml=:@{0}".format(model['matrixnet_id']))
                output_file.write('{0}&relev={1}\n'.format(self._get_request_template(model), param))

        channel.task.mark_resource_ready(r)
        return r.id

    def _get_basesearch_responses(self, videosearch_id, archive_id, queries_id):
        """
            Запуск подзадачи для тестирования
            базового поиска
        """

        self.set_info('Run videosearch with ranking models archive.')
        sub_ctx = {
            'basesearch_type': 'videosearch',
            'executable_resource_id': videosearch_id,
            'config_resource_id': _get_last_released_resource_id(resource_types.VIDEO_SEARCH_CONFIG, arch='any'),
            'search_database_resource_id': _get_priemka_resource_id(resource_types.VIDEO_SEARCH_DATABASE),
            'models_archive_resource_id': archive_id,
            'queries_resource_id': queries_id,
            'get_all_factors': True,
            'recheck_results_n_times': 1,
        }
        sub_task = self.create_subtask(
            task_type='GET_BASESEARCH_RESPONSES',
            input_parameters=sub_ctx,
            description=self.descr,
            arch=ARCH_LINUX,
        )

        return sub_task.id

    def _get_models(self, videosearch_id, archive_id):
        """
            Тестирует целостность моделей архива
            и возвращает список идентификаторов формул
        """

        self.set_info('Retrieve models IDs from archive.')
        # get latest archiver executable
        archiver_resource = _get_priemka_resource_id(resource_types.ARCHIVER_TOOL_EXECUTABLE)
        archiver_path = self.sync_resource(archiver_resource)

        # get list of models from archive
        archive_path = self.sync_resource(archive_id)
        archive_models = get_matrixnets_md5(archiver_path, archive_path)
        self.ctx['formulas'] = sorted(archive_models)

        videosearch_path = self.sync_resource(videosearch_id)
        videosearch_models = get_basesearch_models(videosearch_path, archive_path, old_mode=False)

        # compare models from archive and from videosearch executable
        models = dict((m['matrixnet_name'], m['matrixnet_md5']) for m in videosearch_models if m['matrixnet_memory_source'] == 'dynamic')
        if archive_models != models:
            diff = compare_dicts(archive_models, models)
            del diff['same']
            raise SandboxTaskFailureError('Ranking models in archive and in basesearch are different' % diff)

        # generate list with identifiers of models
        return list(m for m in videosearch_models if m['matrixnet_memory_source'] == 'dynamic')

__Task__ = VideoTestDynamicModelsArchive
