import os
import shutil
import urlparse
from distutils.version import LooseVersion as parse_version

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects.common.environments import SandboxMavenEnvironment
from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import VPS_DIST, VPS_JDK_DIST
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxRadioParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask


class VpsJdkEnviroment(SandboxEnvironment):
    resource_type = VPS_JDK_DIST

    name = 'jdk'

    @property
    def find_resource(self):
        version = parse_version(self.version)
        resources = [resource for release in
                     channel.sandbox.list_releases(self.resource_type, ctm.OSFamily.LINUX, limit=100) for
                     resource in release.resources]
        for resource in resources:
            if parse_version(str(resource.attributes['version'])) >= version:
                self.resource_id = resource.id
                return channel.task.sync_resource(resource)
        raise common.errors.SandboxEnvironmentError("Couldn't find resource for version {}".format(self.version))

    def prepare(self):
        path = self.find_resource
        if not os.path.exists(path):
            shutil.copytree(self.get_environment_resource(), path)
        os.environ['PATH'] = os.path.join(path, 'bin') + ':' + os.environ['PATH']
        os.environ['JAVA_HOME'] = os.path.join(path)
        os.environ['LD_LIBRARY_PATH'] = ':'.join(
            filter(None, [os.path.join(path, 'lib'), os.environ.get('LD_LIBRARY_PATH')]))


class GitVcsRoot(object):
    def __init__(self, url=None, location=None, *args, **kwargs):
        self.url = url
        self.location = location
        super(GitVcsRoot, self).__init__()

    def obtain(self):
        url, rev, location = self.get_url_rev()
        self.location = self.location or location
        if os.path.exists(self.location):
            shutil.rmtree(self.location)
        run_process(['git', 'clone', url, self.location], log_prefix='vcs')
        if rev:
            run_process(['git', 'checkout', rev, '-b', rev], log_prefix='vcs', work_dir=self.location)
        return self

    def get_url_rev(self):
        scheme, netloc, path, query, _ = urlparse.urlsplit(self.url)
        path, rev = '@' in path and path.rsplit('@', 1) or (path, None)
        url = urlparse.urlunsplit((scheme, netloc, path, query, None))
        _, location = '/' in path and path.rsplit('/', 1) or (None, None)
        assert location
        return url, rev, location


class VpsVcsUrlParam(SandboxStringParameter):
    name = 'vps_vcs'
    description = 'Vps Vcs'
    required = True
    default_value = 'https://github.yandex-team.ru/mobile-search/vps.git'


class ViewporteseVcsUrlParam(SandboxStringParameter):
    name = 'viewportese_vcs'
    description = 'Viewportese Vcs'
    required = False


class EnvTypeParam(SandboxRadioParameter):
    name = 'env_type'
    description = 'Environment'
    default_value = 'testing'
    choices = [(v, v) for v in 'production testing development'.split()]


class MvnParam(SandboxStringParameter):
    name = 'mvn_cmd'
    description = 'Additional Maven command line parameters'
    required = False


class NannyParam(SandboxBoolParameter):
    name = 'to_nanny'
    description = 'Release to Nanny'
    default_value = True


# noinspection PyAbstractClass
class VpsBuildTask(SandboxTask, nanny.ReleaseToNannyTask):
    execution_space = 2048

    environment = (
        VpsJdkEnviroment('1.8.0'),
        SandboxMavenEnvironment('3.2.2')
    )

    type = 'VPS_BUILD'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = (
        VpsVcsUrlParam,
        MvnParam,
        EnvTypeParam
    )

    resources = {
        VPS_DIST: 'vps/distribution/target/releases/vpsd'
    }

    runtime_env = 'vps.runtime.environment'

    VPS_PATH = 'vps'

    ver = None
    seq = None
    branch = None
    rev = None

    def on_execute(self):
        self.obtain()
        self.descr = 'r{}.{}.{}.{}.{}'.format(*self.version())
        self.build()
        self.publish()

    def version(self):
        workdir = self.vps()

        self.ver = run_process(
            ['mvn',
             '-q', '-Dexec.executable=echo', '-Dexec.args=\'${project.version}\'', '-N', 'exec:exec'],
            'mvn',
            work_dir=workdir,
            outs_to_pipe=True,
            shell=True
        ).stdout.read().strip()

        self.seq = run_process(
            ['git', 'rev-list', 'HEAD', '|', 'wc', '-l'],
            'git',
            work_dir=workdir,
            outs_to_pipe=True,
            shell=True
        ).stdout.read().strip()

        self.branch = run_process(
            ['git', 'rev-parse', '--abbrev-ref', 'HEAD'],
            'git',
            work_dir=workdir,
            outs_to_pipe=True,
            shell=True
        ).stdout.read().strip()

        self.rev = run_process(
            ['git', 'rev-parse', 'HEAD'],
            'git',
            work_dir=workdir,
            outs_to_pipe=True,
            shell=True
        ).stdout.read().strip()

        return self.ver, self.seq, self.branch, self.rev, self.id

    def obtain(self):
        GitVcsRoot(self.ctx[VpsVcsUrlParam.name], self.vps()).obtain()
        return self

    def build(self):
        params = ['-D%s=%s' % (self.runtime_env, self.ctx[EnvTypeParam.name],),
                  '-Dsandbox.id=%s' % (self.id,)]
        goals = ['package']
        run_process(['mvn', '-B'] + params + self.ctx[MvnParam.name].split() + goals, 'mvn', work_dir=self.vps())
        return self

    def publish(self):
        for name, path in self.resources.iteritems():
            res = self.create_resource(self.descr, path, name, self.arch,
                                       {"version": self.ver, "sequence": self.seq, "branch": self.branch,
                                        "revision": self.rev})
            self.mark_resource_ready(res)
        return self

    def vps(self):
        return self.path(self.VPS_PATH)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)

    @property
    def release_template(self):
        return self.ReleaseTemplate(subject=self.descr)

    def postprocess(self):
        pass


__Task__ = VpsBuildTask
