# coding: utf-8

# $Header$

import json
import logging
import os
import requests
import time

from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

nanny_its_url = 'http://its.yandex-team.ru/v1/history/upper/?limit=64&skip=0&fmt=json'
svn_path = 'arcadia.yandex.ru:/arc/trunk/arcadia/web/report/data/flags/its'
svn_user = 'zomb-sandbox-rw'

# debug overrides
# svn_path = 'arcadia.yandex.ru:/arc/trunk/arcadia/junk/schizophrenia/SERP-46209'
# svn_user = 'schizophrenia'


class Period(SandboxIntegerParameter):
    name = 'period'
    description = "Watch period, seconds (0 to run just once)"
    default_value = 60
    required = True


class WatchSerpItsFlags(SandboxTask):
    type = 'WATCH_SERP_ITS_FLAGS'

    cores = 1
#    cores = 5 # debugging overrides
    description = "Watch for SERP ITS flags changes (SERP-46209)"
    execution_space = 512
    input_parameters = [Period]
    work_dir = 'work_dir'

    def commit_flags(self, flags):
        comments = []
        updated_files = []
        logging.info('Commit flags %s', json.dumps(flags, indent=2, sort_keys=True))
        for k in flags:
            file = os.path.join(self.work_dir, k + '.txt')
            data = flags[k]['user_value']

            if flags[k]['ruchka_id'] in ['disable_flag_sections', 'disable_flag_sections_video']:
                file = os.path.join(self.work_dir, k + '_' + flags[k]['ruchka_id'] + '.txt')
                data = flags[k]['user_value']
                with open(file, 'w') as fh:
                    fh.write(data)
            else:
                file = os.path.join(self.work_dir, k + '.json')
                data = json.loads(flags[k]['user_value'])
                with open(file, 'w') as fh:
                    fh.write(json.dumps(data, indent=2, sort_keys=True))

            status = Arcadia.status(file)
            logging.info('ARCADIA STATUS: %s', status)
            if len(status):
                updated_files.append(os.path.basename(file))
                if status[:1] == '?':
                    logging.info('Adding new file %s', file)
                    Arcadia.add(file)
                comment = "changed {} at {} by {}@".format(
                    k, flags[k]['time'], flags[k]['author']
                )
                comments.append(comment)
                self.info += "{} committed, {}\n".format(
                    os.path.basename(file), comment)
            else:
                logging.info('File %s is not changed', file)
        logging.info('Committing changes for files %s', updated_files)
        comments = ["New ITS flags, (sandbox:{}:{})".format(self.type, self.id)] + comments
        comments.append('SKIP_CHECK')
        comment = '\n'.join(comments)
        Arcadia.commit(self.work_dir, comment, svn_user)

    def get_flags(self):
        nanny_oauth = self.get_vault_data(self.owner, "nanny_oauth")

        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(nanny_oauth)
        session.headers['Content-Type'] = 'application/json'
        logging.info('GET %s', nanny_its_url)
        response = session.get(nanny_its_url)
        changes = response.json()
        flags = {}
        for i in reversed(changes['result']):
            if i.get("ruchka_id") not in ["upper_flags", "upper_flags_images", "upper_flags_video", "disable_flag_sections", "disable_flag_sections_video"]:
                continue
            key = "_".join(i['location_path'])
            flags[key] = i

        return flags

    def on_execute(self):
        self.ctx['start_time'] = time.time()
        self.info = "Watching {}\n".format(nanny_its_url)

        # checkout only once to avoid unnecessary requests to svn
        # task will fail and be restarted by sandbox if dir became outdated (manual commit to directory)
        self.work_dir = Arcadia.checkout(svn_path, self.work_dir)

        while (1):
            flags = self.get_flags()
            self.commit_flags(flags)

            if self.ctx['period'] < 1:
                logging.info('All done, watch disabled, exit.')
                return True

            time_left = int(self.ctx['start_time'] + self.ctx['kill_timeout'] - time.time())

            if time_left < (self.ctx['period'] + 60):  # 60 sec is a reasonable time for fetch and commit
                logging.info('No time for watch left ({}), exit.'.format(time_left))
                return True
            else:
                time.sleep(self.ctx['period'])


__Task__ = WatchSerpItsFlags
