# coding: utf-8

"""
    Проверка элемента на серпе по схеме данных и на качество
"""

import logging
import os
import json
import yaml
import io

import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError

from sandbox.projects import WebResultSchemaCheckerBase as wrsc
from sandbox.projects.common import link_builder as lb


class ErrorLogger(object):
    def __init__(self, path, val):
        name = val.replace('/', '_')
        self.name = name
        self.log_fname = path + "/" + name + ".log"
        self.failed_data_fname = path + "/" + name + "_failed_data.json"
        self.f = open(self.log_fname, "w")
        self.has_errors = False
        self.has_json = False

    def write(self, string):
        self.f.write("{}\n".format(string))
        self.has_errors = True

    def writelines(self, stringarray):
        for s in stringarray:
            self.write("{}".format(s))
        self.has_errors = True

    def writejson(self, jsdata, cnt, query):
        with io.open(self.failed_data_fname, 'a', encoding='utf8') as fdata:
            fdata.write(convert_unicode("QUERY {} '{}'".format(cnt, query)))
            json_str = json.dumps(jsdata, indent=4, ensure_ascii=False)
            fdata.write(unicode(json_str))
            fdata.write(convert_unicode("\n\n"))
            self.has_json = True

    def close(self):
        if not self.f.closed:
            self.f.close()

    def __del__(self):
        self.close()


class WebResultSchemaChecker(wrsc.WebResultSchemaCheckerBase):
    type = 'WEB_RESULT_SCHEMA_CHECKER'
    dns = ctm.DnsType.DNS64

    input_parameters = wrsc.WebResultSchemaCheckerBase.input_parameters

    environment = (
        environments.PipEnvironment('jsonschema'),
        environments.PipEnvironment('yandex-yt'),
    )

    @property
    def footer(self):
        if not self.is_completed():
            return None
        cnt_checked = self.ctx.get('cnt_checked')
        cnt_failed = self.ctx.get('cnt_failed')
        return [{
            'helperName': '',
            'content': "<b>{} queries checked ({} failed)</b>".format(cnt_checked or 0, cnt_failed or 0)
        }]

    def make_email(self):
        subj = "[DATA CHECK] {} FAILED".format(self.ctx.get('element_name') or self.ctx.get('filter_value'))

        cnt_checked = self.ctx.get('cnt_checked')
        cnt_failed = self.ctx.get('cnt_failed')
        statistic = "<b>{} queries checked ({} failed)</b>".format(cnt_checked or 0, cnt_failed or 0)

        text = "\n".join([
            statistic,
            "See details in task {} <br/>".format(lb.task_link(self.id, 0)),
            "Resource with logs: {},".format(lb.resource_link(self.ctx.get("failures", 0))),
            " direct link: <a href=\"https://proxy.sandbox.yandex-team.ru/{}\">log</a>".format(self.ctx.get("failures", 0))
        ])
        return (subj, text)

    def on_execute(self):
        wrsc.WebResultSchemaCheckerBase.on_execute(self)

    def analyze_shootings_result(self, res, cfg):
        if res is None:
            return
        elif isinstance(res, ErrorLogger):
            if res.has_errors:
                resource = self._create_resource(res.name, res.log_fname, 'RESULT_CHECKER_ERRORS')
                self.mark_resource_ready(resource.id)
                self.ctx['failures'] = resource.id
                if res.has_json:
                    jsdump = self._create_resource(res.name, res.failed_data_fname, 'OTHER_RESOURCE')
                    self.mark_resource_ready(jsdump.id)
                self.ctx['failure_emails'] = cfg.get("owners")
                raise SandboxTaskFailureError("There were some validation errors. See log for details.")
        else:
            SandboxTaskUnknownError("Wrong return type for make_shootings")

    def make_shootings(self, cfg):
        if not isinstance(cfg, dict):
            raise wrsc.CheckerException("Element of config is not a dict")

        if "queries" not in cfg:
            raise wrsc.CheckerException("Element of config has no queries info")

        queries = []
        queries.extend(self.get_queries(cfg))

        logs_path = os.path.abspath('.')
        logger = ErrorLogger(logs_path, self.ctx.get('filter_value'))

        jsschema = get_schema(cfg["schema"])

        from projects.common.proxy_wizard import schema_checker as chk
        self.ctx['cnt_checked'] = 0
        self.ctx['cnt_failed'] = 0

        limit = 0
        if "limit" in cfg:
            limit = cfg["limit"]

        for q in queries:
            # Check no more than 'limit' queries
            if limit and self.ctx['cnt_checked'] >= limit:
                break

            qtext = '; '.join(q).strip()
            logging.info("Query text: {}".format(qtext))

            answer = None
            d = None

            try:
                answer = self.get_answer(cfg, q)
            except Exception as ee:
                logger.write("Cannot get answer for query '{}': {}".format(qtext, ee))
                self.ctx['cnt_checked'] += 1
                self.ctx['cnt_failed'] += 1
                continue

            if not answer:
                if cfg.get("required"):
                    logger.write("QUERY '{}' - no data in json_dump".format(qtext))
                    self.ctx['cnt_failed'] += 1
                self.ctx['cnt_checked'] += 1
                continue

            try:
                d = self.filter_answer(answer)
            except Exception as ee:
                self.ctx['cnt_checked'] += 1
                self.ctx['cnt_failed'] += 1
                logger.write("Cannot filter answer for QUERY {} '{}': {}".format(self.ctx['cnt_failed'], qtext, ee))
                logger.writejson(answer, self.ctx['cnt_failed'], qtext)
                continue

            if not d:
                if cfg.get("required"):
                    logger.write("QUERY '{}' - object with {} '{}' is absent".format(qtext, self.ctx.get('filter_type'), self.ctx.get('filter_value')))
                    self.ctx['cnt_failed'] += 1
                self.ctx['cnt_checked'] += 1
                continue

            logging.info("Ready to check schema")
            query_has_errors = False
            try:
                vdc = chk.make_validation(d, jsschema)
            except wrsc.CheckerException as ce:
                self.ctx['cnt_checked'] += 1
                self.ctx['cnt_failed'] += 1
                logger.write("Trouble with validating answer for QUERY {} '{}': {}".format(self.ctx['cnt_failed'], qtext, ce))
                logger.writejson(d, self.ctx['cnt_failed'], qtext)
                continue
            # except Exception as ee:
            #    self.ctx['cnt_checked'] += 1
            #    self.ctx['cnt_failed'] += 1
            #    logger.write("Error while validating data for QUERY {} '{}': {}".format(self.ctx['cnt_failed'], qtext, ee))
            #    logger.writejson(d, self.ctx['cnt_failed'], qtext)
            #    continue

            if len(vdc.schema_errors) > 0 or len(vdc.custom_errors) > 0:
                self.ctx['cnt_failed'] += 1
                query_has_errors = True
                logger.write("QUERY {} '{}' HAS ERRORS".format(self.ctx['cnt_failed'], qtext))

            if len(vdc.schema_errors) > 0:
                logger.write("SCHEMA ERRORS:")
                logger.writelines(vdc.schema_errors)

            if len(vdc.custom_errors) > 0:
                logger.write("DATA ERRORS:")
                logger.writelines(vdc.custom_errors)

            if query_has_errors:
                logger.write("")
                logger.writejson(d, self.ctx['cnt_failed'], qtext)

            self.ctx['cnt_checked'] += 1

        logger.close()

        if logger.has_errors:
            return logger

        return None


def convert_unicode(s):
    if isinstance(s, unicode):
        return s
    else:
        return unicode(s.decode("utf8"))


def get_schema(path):
    return yaml.load(Arcadia.cat(path))


__Task__ = WebResultSchemaChecker
