# coding: utf-8

"""
    Генерация схемы данных для элемента на серпе
"""

import logging
import os
import json

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskUnknownError, SandboxTaskFailureError

from sandbox.projects.common.proxy_wizard import schema_checker as chk

from sandbox.projects import WebResultSchemaCheckerBase as wrsc


class ReqsLimit(parameters.SandboxIntegerParameter):
    name = 'limit'
    description = 'Request num limit (0 - unlimited)'
    default_value = 100


class WebResultSchemaGenerator(wrsc.WebResultSchemaCheckerBase):
    type = 'WEB_RESULT_SCHEMA_GENERATOR'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = wrsc.WebResultSchemaCheckerBase.input_parameters + [ReqsLimit]

    environment = (
        environments.PipEnvironment('yandex-yt'),
    )

    @property
    def footer(self):
        if not self.is_completed():
            return None
        cnt_checked = self.ctx.get('cnt_checked')
        cnt_used = self.ctx.get('cnt_used')
        return [{
            'helperName': '',
            'content': "<b>Total {} queries, used {} answers</b>".format(cnt_checked or 0, cnt_used or 0)
        }]

    def on_failure(self):
        SandboxTask.on_failure(self)

    def on_execute(self):
        wrsc.WebResultSchemaCheckerBase.on_execute(self)

    def analyze_shootings_result(self, sch, cfg):
        if isinstance(sch, dict):
            name = self.ctx.get("filter_value").replace('/', '_')
            fname = os.path.abspath('.') + "/" + name + "_generated.schema.json"
            with open(fname, "w") as fsch:
                fsch.write(json.dumps(sch, indent=4))

            resource = self._create_resource(name, fname, 'OTHER_RESOURCE')
            self.mark_resource_ready(resource.id)
            return True
        else:
            SandboxTaskUnknownError("Generated schema is not a dict")

    def make_shootings(self, cfg):
        if not isinstance(cfg, dict):
            raise wrsc.CheckerException("Element of config is not a dict")

        if "queries" not in cfg:
            raise wrsc.CheckerException("Element of config has no queries info")

        queries = []
        queries.extend(self.get_queries(cfg))

        jsschema = {}
        self.ctx['cnt_checked'] = 0
        self.ctx['cnt_used'] = 0

        limit = self.ctx.get('limit')

        for q in queries:
            qtext = '; '.join(q).strip()
            logging.info("Query text: {}".format(qtext))

            try:
                d = self.filter_answer(self.get_answer(cfg, q))

                if d:
                    logging.info("Successful answer. Generating schema...")
                    tmp_schema = {}
                    chk.create_schema_tree(d, tmp_schema, "", "")
                    jsschema = chk.merge_schema_trees(jsschema, tmp_schema, "", "", False)
                    self.ctx['cnt_used'] += 1

                self.ctx['cnt_checked'] += 1
                logging.info("TOTAL: {}, SUCCESSFUL: {}".format(self.ctx['cnt_checked'], self.ctx['cnt_used']))

                # Check no more than 'limit' queries
                if limit and self.ctx['cnt_checked'] >= limit:
                    break

            except wrsc.CheckerException as ce:
                logging.error("Trouble with answer for query '{}': {}".format(qtext, ce))

            except Exception as ee:
                logging.error("Error for query '{}': {}".format(qtext, ee))

        if self.ctx['cnt_used'] > 0:
            chk.make_required(jsschema, self.ctx['cnt_used'])
        else:
            SandboxTaskFailureError("No queries was suitable for generating jsonschema")

        return jsschema


__Task__ = WebResultSchemaGenerator
