# coding: utf-8

import logging

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.common import rest

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.proxy_wizard.provider import ProxyTypes
from sandbox.projects.common.differ import coloring

"""
Params:
- proxy revision -> build_wizard_proxy
- config path
- answers name
- 'golden' answers resource id [optional] -> if no, it find last released answers + canon
    -> find appropriate queries (from task)
"""

SEPARATOR = "-" * 10


class ProxyRevision(parameters.SandboxIntegerParameter):
    name = 'proxy_revision'
    description = 'ProxyWizard arcadia revision'
    required = False


class ProxyBuild(parameters.SandboxIntegerParameter):
    name = 'proxy_build'
    description = 'ProxyWizard build resource_id'
    required = False


class ConfigPath(parameters.SandboxArcadiaUrlParameter):
    name = 'config_path'
    description = 'Path to config folder'
    default_value = 'arcadia:/arc/trunk/arcadia/extsearch/wizards/fastres/config/'
    required = True


class AnswersLabel(parameters.SandboxSelectParameter):
    name = 'answers_name'
    description = 'Answers label'
    choices = [(k, k) for k in ProxyTypes.proxies.keys()]
    required = True


class ComparedAnswers(parameters.LastReleasedResource):
    name = 'compared_answers'
    description = 'Proxy answers resource to compare with'
    resource_type = [resource_types.PROXY_WIZARD_ANSWERS]
    required = False


class State:
    Initial = 0
    ProxyBuilding = 1
    ProxyBuilded = 2
    AnswersGenerating = 3
    AnswersCompared = 4


class WizardProxyCompareReqRes(SandboxTask):
    type = 'WIZARD_PROXY_COMPARE_REQRES'
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = (ProxyRevision, ProxyBuild, ConfigPath, AnswersLabel, ComparedAnswers)
    generate_html_diff = False
    new_answers_resource = None
    old_answers_resource = None

    @property
    def footer(self):
        if not self.is_completed():
            return None
        no_diff = self.ctx.get('no_diff')
        if no_diff:
            return [{
                'helperName': '',
                'content': "<b style='color:{}'>No diff</b>".format(coloring.DiffColors.good)
            }]
        else:
            client = rest.Client()
            return client.task[self.ctx['compare_subtask_id']].custom.footer.read()

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        self.ctx['state'] = State.Initial

    def on_execute(self):
        if self.ctx['state'] == State.Initial:
            if not self.ctx[ProxyBuild.name]:
                self._build_proxy_binary()
            else:
                self._set_state(State.ProxyBuilded)

        if self.ctx['state'] == State.ProxyBuilding:
            logging.info('Get proxy binary (resource)')
            self._get_proxy_resource()

        if self.ctx['state'] == State.ProxyBuilded:
            logging.info('Get canonical data')
            self._get_canonical_data()
            logging.info('Generate new answers')
            self._generate_new_answers()

        if self.ctx['state'] == State.AnswersGenerating:
            logging.info('Sync comparing resources')
            self._get_answers_resources()
            logging.info('Comparing...')
            self._compare_binary_responses()

        if self.ctx['state'] == State.AnswersCompared:
            logging.info('Answers compared')
            self._create_verdict()

    def _set_state(self, state):
        self.ctx['state'] = state

    def _get_canonical_data(self):
        compared_task_id = 0

        if not self.ctx[ComparedAnswers.name]:
            compared_resources = channel.sandbox.list_resources(
                resource_type=resource_types.PROXY_WIZARD_ANSWERS,
                omit_failed=True,
                status='READY',
                all_attrs={'ttl': 'inf', 'wiztype': self.ctx[AnswersLabel.name]},
                limit=1
            )

            if not compared_resources or not compared_resources[0].is_ok:
                raise SandboxTaskFailureError('Cannot find proxy answers for comparing')

            compared_resource = compared_resources[0]
            self.ctx[ComparedAnswers.name] = compared_resource.id
            compared_task_id = compared_resource.task_id
        else:
            compared_resource = channel.sandbox.get_resource(self.ctx[ComparedAnswers.name])
            if compared_resource and compared_resource.is_ok:
                compared_task_id = compared_resource.task_id
            else:
                raise SandboxTaskFailureError('Incorrect canon resource ID')

        logging.info('Selected compared res: {}, task: {}'.format(self.ctx[ComparedAnswers.name], compared_task_id))

        compared_task = channel.sandbox.get_task(compared_task_id)
        if 'canon_answers' in compared_task.ctx:
            self.ctx['canonical_answers_id'] = compared_task.ctx['canon_answers']
        else:
            canon_resources = channel.sandbox.list_resources(resource_type='PROXY_WIZARD_CANON_ANSWERS', task_id=compared_task_id, limit=1)
            logging.info('Sync comparing resources')
            if not canon_resources or not canon_resources[0].is_ready():
                raise SandboxTaskUnknownError('No canon answers in compared task')
            self.ctx['canonical_answers_id'] = canon_resources[0].id

        if 'canon_build_id' in compared_task.ctx:
            self.ctx['canon_build_id'] = compared_task.ctx['canon_build_id']
        else:
            raise SandboxTaskUnknownError('Cannot get canon_build_id from compared task')

        if 'queries_resource_id' in compared_task.ctx:
            self.ctx['requests_id'] = compared_task.ctx['queries_resource_id']
        else:
            raise SandboxTaskUnknownError('Cannot get canon_build_id or requests_id from compared task')

    def _build_proxy_binary(self):
        svn_revision = self.ctx[ProxyRevision.name]
        if not svn_revision:
            raise SandboxTaskFailureError('ProxyWizard resource id or svn revision must be specified')

        logging.info('Create build task')
        build_input_parameters = {}
        for k in ProxyTypes.proxies.keys():
            build_input_parameters['build_' + k] = True if k == self.ctx[AnswersLabel.name] else False

        for k in ProxyTypes.others.keys():
            build_input_parameters['build_' + k] = False

        build_input_parameters['checkout_arcadia_from_url'] = 'arcadia:/arc/trunk/arcadia@{}'.format(svn_revision)

        build_subtask = self.create_subtask(
            'BUILD_WIZARD_PROXY',
            'Build {} from revision {}'.format(self.ctx[AnswersLabel.name], svn_revision),
            input_parameters=build_input_parameters,
            arch=self.arch
        )

        logging.info('Task BuildWizardProxy created, waiting...')
        self._set_state(State.ProxyBuilding)
        self.ctx['build_subtask_id'] = build_subtask.id
        self.wait_task_completed(build_subtask)

    def _get_proxy_resource(self):
        logging.info('Build task finished, getting resource')
        if self.ctx['build_subtask_id'] is None:
            raise SandboxTaskUnknownError('Cannot get resource from undefined build task')
        # TODO: use dict with resource types
        build_resources = channel.sandbox.list_resources(
            resource_type=ProxyTypes.proxies[self.ctx[AnswersLabel.name]][0],
            omit_failed=True,
            task_id=self.ctx['build_subtask_id']
        )
        if build_resources:
            self.ctx[ProxyBuild.name] = build_resources[0].id
            logging.info('Builded resource {}'.format(build_resources[0].id))
            self._set_state(State.ProxyBuilded)
        else:
            raise SandboxTaskUnknownError('Cannot get proxy_executable from build task')

    def _generate_new_answers(self):
        logging.info('Create ReqRes task')
        reqres_subtask = self.create_subtask(
            'WIZARD_PROXY_ANSWERS_REQRES',
            'Answers for proxy "{}" build {}'.format(self.ctx[AnswersLabel.name], self.ctx[ProxyBuild.name]),
            input_parameters={
                'proxy_build_id': self.ctx[ProxyBuild.name],
                'canon_build_id': self.ctx['canon_build_id'],
                'config_path': self.ctx[ConfigPath.name],
                'canon_answers': self.ctx['canonical_answers_id'],
                'queries_resource_id': self.ctx['requests_id'],
                'answers_name': self.ctx[AnswersLabel.name]
            },
            arch=self.arch
        )
        logging.info('ReqRes task created ({}), waiting...'.format(reqres_subtask.id))
        self.ctx['reqres_subtask_id'] = reqres_subtask.id
        self._set_state(State.AnswersGenerating)
        self.wait_task_completed(reqres_subtask)

    def _get_answers_resources(self):
        logging.info('ReqRes task finished, getting resource')
        if self.ctx['reqres_subtask_id'] is None:
            raise SandboxTaskUnknownError('Cannot get resource from undefined reqres task')

        answers_resources = channel.sandbox.list_resources(
            resource_type=resource_types.PROXY_WIZARD_ANSWERS,
            omit_failed=True,
            task_id=self.ctx['reqres_subtask_id']
        )
        if not answers_resources:
            raise SandboxTaskUnknownError('Cannot get reqres answers from build task')

        self.new_answers_resource = channel.sandbox.get_resource(answers_resources[0].id)
        self.old_answers_resource = channel.sandbox.get_resource(self.ctx[ComparedAnswers.name])
        self.requests_resource = channel.sandbox.get_resource(self.ctx['requests_id'])

    def _compare_binary_responses(self):
        if self.old_answers_resource.file_md5 == self.new_answers_resource.file_md5:
            self.ctx["no_diff"] = True
            logging.info('No md5 diff, return')
            return
        compare_input_parameters = {
            'basesearch_responses1_resource_id': self.old_answers_resource.id,
            'basesearch_responses2_resource_id': self.new_answers_resource.id
        }
        logging.info('Create comparing subtask')
        compare_subtask = self.create_subtask(
            'COMPARE_BINARY_RESPONSES',
            'Compare responces from \'{}\' wizard_proxies'.format(self.ctx[AnswersLabel.name]),
            input_parameters=compare_input_parameters,
            arch=self.arch
        )

        self.ctx['compare_subtask_id'] = compare_subtask.id
        logging.info('Task CompareBinaryResponses id={} created, waiting...'.format(compare_subtask.id))
        self._set_state(State.AnswersCompared)
        self.wait_task_completed(compare_subtask)

    def _create_verdict(self):
        logging.info('Comparing task finished, getting resource')
        if self.ctx['compare_subtask_id'] is None:
            raise SandboxTaskUnknownError('Undefined comparing task')

        subtask = channel.sandbox.get_task(self.ctx['compare_subtask_id'])
        if not subtask:
            raise SandboxTaskUnknownError('Cannot get child task {}'.format(self.ctx['compare_subtask_id']))

        self.ctx['no_diff'] = subtask.ctx['compare_result']

        if not subtask.ctx['compare_result']:
            # compare_result == True -> no diff
            comparing_resources = channel.sandbox.list_resources(
                resource_type=resource_types.BASESEARCH_RESPONSES_COMPARE_RESULT,
                omit_failed=True,
                task_id=self.ctx['compare_subtask_id']
            )
            if not comparing_resources:
                raise SandboxTaskUnknownError('Cannot get diff resource from child comparing task')

            self.ctx['diff_resource_id'] = comparing_resources[0].id


__Task__ = WizardProxyCompareReqRes
