# coding: utf-8

import json
import logging
import os
import struct
import time
import urllib2

import jinja2
import sandbox.sandboxsdk.util as sdk_util
import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common.proxy_wizard import executable_types as ex_types
from sandbox.projects.common.proxy_wizard.provider import construct_proxy_provider, ProxyTypes
from sandbox.projects.common.wizard.minigun import MiniGun


class ProxyBuild(parameters.SandboxIntegerParameter):
    # We cannot use Resource selector because we don't know exact type of proxy
    name = 'proxy_build_id'
    description = 'ProxyWizard resource ID'
    required = True


class CanonBuild(parameters.LastReleasedResource):
    name = 'canon_build_id'
    description = 'CanonWizard resource ID'
    resource_type = [ex_types.PROXY_WIZARD_CANON_EXECUTABLE]


class ConfigPath(parameters.SandboxArcadiaUrlParameter):
    name = 'config_path'
    description = 'Path to config folder'
    default_value = 'arcadia:/arc/trunk/arcadia/extsearch/wizards/fastres/config/'
    required = True


class ReqsToShoot(parameters.ResourceSelector):
    name = 'queries_resource_id'
    description = 'Requests for shooting'
    resource_type = [resource_types.PROXY_WIZARD_PLAIN_QUERIES]
    required = True


class ReqsLimit(parameters.SandboxIntegerParameter):
    name = 'limit'
    description = 'Request num limit (0 - unlimit)'
    default_value = 0


class AnswersName(parameters.SandboxSelectParameter):
    name = 'answers_name'
    description = 'Answers name'
    required = True
    choices = [(k, k) for k in ProxyTypes.proxies.keys()]


class WizardProxyGenerateReqRes(SandboxTask):
    type = 'WIZARD_PROXY_GENERATE_REQRES'

    input_parameters = (ProxyBuild, CanonBuild, ConfigPath, ReqsToShoot, ReqsLimit, AnswersName)
    client_tags = ctc.Tag.Group.LINUX

    proxy_port = 19245
    canon_port = 19970
    storage_file = 'canon.stor'
    answers_file = 'answers.bin'
    config_path = 'config'

    THREADS_NUM = 5

    def _prepare_data(self):
        config_path = Arcadia.checkout(self.ctx[ConfigPath.name], self.config_path)
        json_path = os.path.join(config_path, 'config_data.json')
        if not os.path.exists(json_path):
            raise SandboxTaskFailureError('config_data.json is absent')

        jsconf = json.load(open(json_path))
        for r in jsconf["Resources"]:
            resource_path = ''
            if isinstance(r["ID"], (str, unicode)) and r["ID"] == 'LRR':
                logging.info('resource id is string: {}, type is {}'.format(r["ID"], r["Type"]))
                logging.info('Getting list of resources')
                tasks = channel.sandbox.list_releases(resource_type=r["Type"], arch=sdk_util.system_info()['arch'], limit=1)
                if not tasks:
                    raise SandboxTaskFailureError('No tasks with released resources of type {}'.format(r["Type"]))
                logging.info('Resource release task id {}'.format(tasks[0].id))
                resource = apihelpers.list_task_resources(tasks[0].id, r["Type"], sdk_util.system_info()['arch'])
                if not resource:
                    raise SandboxTaskFailureError('Cannot get %s from task #%s' % (r["Type"], tasks[0].id))
                resource_path = channel.task.sync_resource(resource[0].id)
            else:
                try:
                    resource_id = int(r["ID"])
                except ValueError:
                    raise SandboxTaskUnknownError('Incorrect format of resource_id for {}: {}'.format(r.get("Type"), r["ID"]))
                logging.info('Resource id is number: {}, type is {}'.format(resource_id, r.get("Type")))
                resource_path = channel.task.sync_resource(resource_id)

            if resource_path == '':
                raise SandboxTaskUnknownError('Cannot sync resource')

            logging.info('Resource path: {}'.format(resource_path))
            if not os.path.exists(r["File"]):
                os.symlink(resource_path, r["File"])

        if not os.path.exists(os.path.join(self.config_path, 'config.tpl')):
            raise SandboxTaskFailureError('config.tpl is absent')

        if not os.path.exists('config.tpl'):
            os.symlink(os.path.join(self.config_path, 'config.tpl'), 'config.tpl')

        self._create_configs(jsconf)

    def _create_links_to_canon(self, port, name):
        return 'http://localhost:{}/set/{}/?'.format(port, name)

    def _create_configs(self, conf):
        template = get_template('config.tpl')

        conf["ConfigPath"] = os.path.join(os.path.abspath('.'), self.config_path)

        conf["Port"] = self.proxy_port
        conf["Host"] = 'localhost'
        conf["Eventlog"] = '/tmp/proxy_proxy'

        for s in conf["Sources"]:
            s["OrigUrl"] = s["Url"]
            s["Url"] = self._create_links_to_canon(self.canon_port, s["Name"])

        config = template.render(data=conf)
        with open('proxyconf.cfg', 'w') as f:
            f.write(config.encode('utf-8'))

        self.proxy_config = 'proxyconf.cfg'
        self.canon_configs = []

        canon_tmpl = jinja2.Template(Arcadia.cat('arcadia:/arc/trunk/arcadia/extsearch/wizards/canon/config/config.tpl'))

        canon = {
            "Host": "localhost",
            "Port": self.canon_port,
            "Storage": self.storage_file,
            "Eventlog": '/tmp/canon_eventlog',
            "Sources": []
        }

        for s in conf["Sources"]:
            canon["Sources"].append({
                "Name": s["Name"],
                "Url": s["OrigUrl"]
            })

        canon_config_text = canon_tmpl.render(canon=canon)
        with open('canon.cfg', 'w') as f:
            f.write(canon_config_text.encode('utf-8'))

        self.canon_config = 'canon.cfg'

    def _create_storage_resource(self):
        if os.path.exists(self.storage_file):
            logging.info('Save canon storage file')
            self._create_resource('Canon answers', self.storage_file, 'PROXY_WIZARD_CANON_ANSWERS', attrs=self.res_attributes)
        else:
            raise SandboxTaskFailureError('Storage file was not created')

    def on_execute(self):
        self._prepare_data()
        self.res_attributes = {
            'wiztype': self.ctx[AnswersName.name]
        }

        requests_file = self.sync_resource(self.ctx[ReqsToShoot.name])
        requests = [line.strip() for line in open(requests_file)]
        if self.ctx.get(ReqsLimit.name):
            limit = int(self.ctx[ReqsLimit.name])
            requests = requests[:limit]

        logging.info('Create proxy')
        with construct_proxy_provider(self.ctx[ProxyBuild.name], self.proxy_config, self.proxy_port) as proxy_wizard:
            logging.info('Create canons')
            with construct_proxy_provider(self.ctx[CanonBuild.name], self.canon_config, self.canon_port) as proxy_canon:

                proxy_address = '{proxy.host}:{proxy.port}'.format(proxy=proxy_wizard)
                threads = self.THREADS_NUM
                if len(requests) < self.THREADS_NUM * 10:
                    threads = 1
                minigun = MiniGun(proxy_address, requests, save_answers=True, threads=threads, attempts_per_req=1, with_newlines=True)

                time.sleep(5)
                if not proxy_wizard.alive():
                    raise SandboxTaskFailureError('proxy_wizard is dead')
                if not proxy_canon.alive():
                    raise SandboxTaskFailureError('proxy_canon is dead')

                time.sleep(5)

                logging.info('Ping {proxy.host}:{proxy.port}'.format(proxy=proxy_canon))
                url = 'http://{proxy.host}:{proxy.port}/admin?action=ping'.format(proxy=proxy_canon)
                answer = urllib2.urlopen(url, timeout=5).read()
                logging.info('Ping answer is: {}'.format(answer))

                minigun.shoot()

                with open(self.answers_file, 'w') as f:
                    logging.info('Write proxy answers to file')
                    for answ in minigun.answers:
                        str_size = struct.pack('i', len(answ))
                        f.write(str_size)
                        f.write(answ)
                    logging.info('Create resource with answers')
                    self._create_resource(self.ctx[AnswersName.name], self.answers_file, 'PROXY_WIZARD_ANSWERS', attrs=self.res_attributes)
                    logging.info('Answers saved')

        time.sleep(1)
        logging.info('All stopped')

        self._create_storage_resource()


def get_template(template_filename):
    template_path = os.path.abspath('.')
    logging.info('Template path: {}'.format(template_path))
    env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
    return env.get_template(template_filename)


__Task__ = WizardProxyGenerateReqRes
