# coding: utf-8

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.wizard import parameters as wp
from sandbox.projects.common.wizard.minigun import MiniGun
from sandbox.projects.common.wizard.providers import construct_wizard_provider


class WizardBuild(wp.WizardBuildOrRemoteParameter):
    name = 'wizard_build'
    description = 'Wizard build id or remote wizard address (host:port)'


class WizardRuntime(wp.WizardRuntimeParameter):
    name = 'runtime_id'
    description = 'Wizard runtime id (not required)'


class ReqsLimit(parameters.SandboxIntegerParameter):
    name = 'limit'
    description = 'Maximum number of requests to send (0 = unlimited)'


class WizardResponses(SandboxTask):
    type = 'WIZARD_RESPONSES'
    input_parameters = (WizardBuild, WizardRuntime, wp.Queries, ReqsLimit, wp.ExtraQueryParams)
    client_tags = ctc.Tag.Group.LINUX

    def on_execute(self):
        with construct_wizard_provider(self.ctx[WizardBuild.name], self.ctx.get(WizardRuntime.name)) as wizard:
            limit = utils.get_or_default(self.ctx, ReqsLimit) or None
            query = utils.get_or_default(self.ctx, wp.ExtraQueryParams)
            with open(self.sync_resource(self.ctx[wp.Queries.name])) as fd:
                requests = [line.strip() for line in fd][:limit]
            requests = [r + ('&' if '?' in r else '?') + (query or '').lstrip('&') + '&waitall=da' for r in requests]

            # MiniGun will drop the last `len(requests) % threads` paths.
            threads = min(len(requests), 10)
            wizard_address = '{}:{}'.format(wizard.host, wizard.port)
            minigun = MiniGun(wizard_address, requests, save_answers=True, threads=threads, attempts_per_req=1)
            minigun.shoot()

            with open('answers.txt', 'w') as answers:
                answers.write('\n'.join(minigun.answers))
            self.create_resource('wizard answers', 'answers.txt', resource_types.PLAIN_TEXT_QUERIES)

            if not wizard.alive():
                raise SandboxTaskFailureError('wizard is dead')


__Task__ = WizardResponses
