# -*- coding: utf-8 -*-

from __future__ import print_function

import datetime
import json
import logging
import os
import shutil
import textwrap
import threading

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel

from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag

from sandbox.projects import resource_types
from sandbox.projects.common import constants
# from sandbox.projects.common.build.parameters import BuildSystem  # SPI-5746
from sandbox.projects.common import utils
from sandbox.projects.common.arcadia import sdk as arc_sdk
from sandbox.projects.common.wizard.wizard_builder import WizardBuilder
from sandbox.projects.common.wizard.utils import setup_hosts_sdk1
from sandbox.projects.common.wizard.exception_manager import ExceptionManager
from sandbox.projects.websearch.begemot import AllBegemotServices
from sandbox.projects.websearch.begemot.common import Begemots
from sandbox.projects.websearch.begemot.tasks.BegemotYT.common import CypressShardUpdater
from . import merge

from sandbox.projects.WizardRuntimeBuild.ya_make import YaMake

YaMake = YaMake.YaMake


class RuntimeDataSource(parameters.SandboxStringParameter):
    '''
    usually arcadia:/arc/trunk/arcadia/search/wizard/data/fresh/
    or legacy: arcadia:/robots/trunk/wizard-data/ .
    May be a branch url.
    '''
    Robots = 'arcadia:/robots/trunk/wizard-data/'
    ArcadiaRoot = 'arcadia:/arc/trunk/arcadia'
    Arcadia = ArcadiaRoot + '/search/wizard/data/fresh/'

    name = 'runtime_data_source'
    description = 'Runtime data source'
    default_value = Arcadia
    required = False


class RuntimeLegacy(parameters.SandboxStringParameter):
    '''
    If merging, the location to merge from. Usually
    arcadia:/robots/trunk/wizard-data/
    '''
    name = 'runtime_legacy'
    description = 'Runtime legacy url'
    default_value = RuntimeDataSource.Robots
    required = False


class MergeOption(parameters.SandboxRadioParameter):
    '''This option should be deleted soon'''
    NoMerge = 'nomerge'
    Merge = 'merge'
    OnlyMerge = 'only_merge'

    name = "merge"
    description = "Merge legacy runtime to Arcadia? (REQWIZARD-965)"
    choices = [
        ('Do not merge anything, use Arcadia only', NoMerge),
        ('Merge from "Runtime legacy url" to "Runtime data source"', Merge),
        ('Merge and exit, do not build', OnlyMerge),
    ]
    default_value = NoMerge


class IgnoreSandboxConflicts(parameters.SandboxBoolParameter):
    '''
        Use when sandbox compression method changed, no md5sums match,
        and you are sure you won't lose any changes from /arc
    '''
    name = "sandbox_ignore_conflicts"
    description = 'Always overwrite sandboxed resources while merging'
    default_value = False


class DoMerge(parameters.SandboxBoolParameter):
    '''
        Merge the latest commits from /robots.
        This is done automatically every 3 hours,
        usually you do not need to enable this option manually.
        https://wiki.yandex-team.ru/begemot/fresh/#avtomjorzhilka
    '''
    name = "do_merge"
    description = 'Merge from a legacy repo to Arcadia and Sandbox'
    default_value = False


class DoCommit(parameters.SandboxBoolParameter):
    '''
        Disable if you're getting weird patches and debugging them.
        If merging, this option is required to build.
    '''
    name = "do_commit"
    description = 'Commit. Required to build if merging. Does nothing if not merging.'
    default_value = False


class DoBuild(parameters.SandboxBoolParameter):
    '''
        Build shards?
    '''
    name = "do_build"
    description = 'Build. Note that if merging, you also have to commit.'
    default_value = True


class VerboseLogs(parameters.SandboxBoolParameter):
    '''
        Produce very verbose "merge.log"
    '''
    name = "want_verbose_logs"
    description = 'More information for debugging merges'
    default_value = False


class BuildForProduction(parameters.SandboxBoolParameter):
    '''
        Create begemot packs based on production-branch shards
        instead of current shards from Arcadia.
        Set this to False in per-commit checks,
        set this to true for sandbox release scheduler.
    '''
    name = "production_build"
    description = "Build for production binary and shards"
    default_value = True


class SandboxValidationMode(parameters.SandboxBoolParameter):
    '''
        Light mode for testing new Sandbox releases (SANDBOX-5077).
        Does nothing currently.
    '''
    name = "sandbox_validation"
    description = "No-op, only sandbox testing"
    default_value = False


class RobotsRevision(parameters.SandboxIntegerParameter):
    '''default: latest'''
    name = 'runtime_data'
    description = 'Runtime data revision in the /robots repository'


class ArcadiaRevision(parameters.SandboxIntegerParameter):
    '''default: latest'''
    name = 'arcadia_revision'
    description = 'Arcadia revision to checkout'
    default = None


class ParentResources(parameters.DictRepeater, parameters.SandboxStringParameter):
    # This parameter is used by BUILD_WIZARD_2 and is not visible in the UI.
    name = 'parent_resources'
    description = 'Transfer resources to parent task (resource_type, resource_id)'


class BegemotShards(parameters.SandboxBoolGroupParameter):
    name = 'begemot_shards'
    description = 'Begemot shards to build'
    choices = [
        (sname, sname) for sname, s in Begemots
        if s.fresh_resource_name is not None
    ]
    default_value = ' '.join(i[0] for i in choices if Begemots[i[0]].release_fresh)


class BegemotShardBuilder:
    def __init__(self, service, resource_ids_dict):
        self.service = service

        self._resource_ids_dict = resource_ids_dict
        self.path = os.path.join('result_dir', service, 'fresh')

    def set_static_data_dir(self, dir):
        self._static_data_yamake_path = os.path.join(dir, 'ya.make')

    def build_from_shard(self, shard_path, revision, task_id):
        for item in self._rules_in_yamake(self._static_data_yamake_path):
            logging.debug("Put item {} to shard {}".format(item, shard_path))
            shard_item_path = os.path.join(shard_path, item)
            if os.path.isdir(shard_item_path):
                shutil.copytree(shard_item_path, os.path.join(self.path, item))
        now = datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ')
        with open(os.path.join(self.path, 'version.pb.txt'), 'w') as version_pb_file:
            version_pb_file.write('GenerationTime: "{}"\nRevision: {}\n Task: {}, ShardName: "{}"\n'.format(now, revision, task_id, self.service))
        process.run_process(['tar', '-cf', '../fresh.tar', '.'], check=True, wait=True, work_dir=self.path)

    def create_resource(self, task):
        r = task.create_resource(
            task.descr,
            self.path,
            AllBegemotServices.Service[self.service].fresh_data_resource_type
        )
        self._resource_ids_dict[self.service] = r.id
        p = task.create_resource(
            task.descr,
            self.path + '.tar',
            AllBegemotServices.Service[self.service].fresh_data_resource_packed_type
        )
        self._resource_ids_dict[self.service + 'packed'] = p.id

    @property
    def packed_resource_id(self):
        return self._resource_ids_dict[self.service + 'packed']

    @property
    def resource_id(self):
        return self._resource_ids_dict[self.service]

    @staticmethod
    def _rules_in_yamake(yamake_path):
        if not os.path.exists(yamake_path):
            # A shard has not yet appeared in any released branch
            return []
        shard = YaMake(yamake_path)
        root = 'search/wizard/data/wizard/'
        try:
            return [
                r.split(root, 1)[1].split('#', 1)[0].strip()
                for r in shard.peerdir
                if r and not r.startswith('#')
            ]
        except Exception as x:
            raise TaskFailure('Failed processing {yamake_path},\nthe line "{r}" does not contain "{root}"\n{x}'.format(**locals()))


class WizardRuntimeBuild(task.SandboxTask, CypressShardUpdater):
    type = 'WIZARD_RUNTIME_BUILD'
    execution_space = 250 * 1024  # MB
    input_parameters = [
        RuntimeDataSource, BuildForProduction,
        ArcadiaRevision, RuntimeLegacy, RobotsRevision,
        DoMerge, DoCommit, DoBuild, BegemotShards, VerboseLogs,
        IgnoreSandboxConflicts, SandboxValidationMode,
        # BuildSystem,  # SPI-5746
    ] + CypressShardUpdater.input_parameters
    # Here we need to list all possible tags and later refine them in on_enqueue.
    # That's the only possible way in SDK1
    client_tags = Tag.GENERIC | Tag.CUSTOM_WIZARD | Tag.CUSTOM_BEGEMOT_RELEASES

    wizard_runtime_path = 'wizard.runtime'  # must match Nanny service configuration
    packed = resource_types.WIZARD_RUNTIME_PACKAGE
    unpacked = resource_types.WIZARD_RUNTIME_PACKAGE_UNPACKED
    logger = logging.getLogger(type)
    logger.setLevel(logging.DEBUG)

    def _make_begemot_shard_builders(self):
        if 'begemot_shard_resource_ids' not in self.ctx:
            self.ctx['begemot_shard_resource_ids'] = dict()
        return [
            BegemotShardBuilder(s, self.ctx['begemot_shard_resource_ids'])
            for s in utils.get_or_default(self.ctx, BegemotShards).split()
        ]

    def pre_on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        setup_hosts_sdk1(self)
        merge_required = utils.get_or_default(self.ctx, DoMerge) or utils.get_or_default(self.ctx,
                                                                                         MergeOption) != MergeOption.NoMerge
        commit_required = utils.get_or_default(self.ctx, DoCommit)
        build_required = utils.get_or_default(self.ctx, DoBuild)

        self.ctx[DoMerge.name] = merge_required
        self.ctx[DoCommit.name] = commit_required
        self.ctx[DoBuild.name] = build_required

        if merge_required:
            try:
                self.tags.append('merge')
            except Exception:
                self.tags = ['merge']

        if not build_required:
            return

        if build_required and merge_required and not commit_required:
            self.info += 'Build disabled: when merging, you have to commit to build ' \
                         '(otherwise it is too tedious to generate correct ya make-s ' \
                         'with sandbox resources and later rollback resources)'
            return

        self.ctx['out_resource_id'] = self.create_resource(self.descr, self.wizard_runtime_path + '.tar',
                                                           self.packed).id
        self.ctx['unpacked_resource_id'] = self.create_resource(self.descr, self.wizard_runtime_path, self.unpacked).id

        for b in self._make_begemot_shard_builders():
            b.create_resource(self)

    def on_enqueue(self):
        self.pre_on_enqueue()
        try:
            tags = self.ctx.get(constants.SANDBOX_TAGS, "").strip().upper()
            self.client_tags = self.__class__.client_tags & Tag.Query(tags)
        except:
            logging.exception("Error in setting client tags on_enqueue of task #%s type %s", self.id, self.type)

    def checkout_arcadia(self, arcadia_root, svn_dirs, path):
        revision = self.ctx[ArcadiaRevision.name]
        Arcadia.checkout(arcadia_root, path, revision=revision, depth='immediates')
        for svndir in svn_dirs:
            paths.make_folder(os.path.join(path, svndir))
            Arcadia.update(os.path.join(path, svndir), revision=revision, parents=True, set_depth='infinity')

    def checkout_legacy_runtime(self, legacy_fresh_results):
        assert self.merge_required, 'This is a bug, contact whoever is to blame. This function makes no sense when not merging.'
        url = utils.get_or_default(self.ctx, RuntimeLegacy)
        if url:
            revision = int(self.ctx[RobotsRevision.name])
            url = Arcadia.replace(url, revision=revision)
            legacy_fresh_results['path'] = path = Arcadia.get_arcadia_src_dir(url)
        if not url or not path:
            raise ValueError('Failed to checkout legacy_runtime: perhaps incorrect url or runtime?\n' +
                             'local_path: None, url: %s, revision: %s' % (url, revision))

    def apply_patch(self, path):
        '''Apply patch to a built fresh for testing runtime data build (see WizardRuntimeBuildPatched)'''
        pass

    def apply_arcadia_patch(self, arc_path):
        '''Apply patch to Arcadia sources for testing runtime data build (see WizardRuntimeBuildPatched)'''
        pass

    def rewrite_version_info(self, path):
        with open(path) as fp:
            v = json.load(fp)
        v['GenerationTime'] = datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ')
        v['Revision'] = self.ctx[ArcadiaRevision.name]
        v['RobotsRevision'] = self.ctx[RobotsRevision.name]
        v['Task'] = self.id
        with open(path, 'w') as fp:
            json.dump(v, fp)

    def make_runtime_packages(self, pre_build_path, begemot_shard_builders):
        attrs = {'runtime_data': str(self.ctx[RobotsRevision.name])}
        tar = channel.sandbox.get_resource(self.ctx['out_resource_id']).path
        self.rewrite_version_info(os.path.join(pre_build_path, 'version.info'))
        shutil.copytree(pre_build_path, self.wizard_runtime_path)  # wizard fresh
        process.run_process(['tar', '-cf', tar, self.wizard_runtime_path], check=True, wait=True, log_prefix='tar')
        for b in begemot_shard_builders:
            b.build_from_shard(pre_build_path, self.ctx[ArcadiaRevision.name], self.id)
            utils.set_resource_attributes(b.resource_id, attrs)
            utils.set_resource_attributes(b.packed_resource_id, attrs)
        utils.set_resource_attributes(self.ctx['out_resource_id'], attrs)
        utils.set_resource_attributes(self.ctx['unpacked_resource_id'], attrs)

    def validate_build_results(self, sources, build_output):
        needed = []
        for d in os.listdir(sources):
            if d == 'package' or not os.path.isdir(os.path.join(sources, d)):
                continue
            try:
                yamake = YaMake(os.path.join(sources, d, 'ya.make'))
                for var in yamake.set:
                    if len(var) == 2 and var[0] == 'RESULT':
                        built = os.path.basename(var[1])
                        break
                else:
                    built = d
            except Exception:
                built = d
            needed.append(built)

        missing = [n for n in needed if not os.path.exists(os.path.join(build_output, n))]
        if missing:
            for dirs in ['sources', 'build_output']:
                logging.info('%s dir: %s\n%s\n\n' % (dirs, locals()[dirs], '\n'.join(sorted(os.listdir(locals()[dirs])))))
            raise TaskFailure('\n'.join([
                'Ya make produced incorrect results, '
                'the following files/directories are missing in the build results:'
            ] + sorted(missing) + [
                '', 'I will stop here to avoid another SPI-5746'
            ]))
        else:
            logging.info('\n'.join(
                ['The following dirs were found in the build output as expected after SPI-5764:'] + sorted(needed)
            ))
            self.info += 'Build result validated without errors (see common.log for details)'

    def _checkout_arcadia_branch(self):
        arc = WizardBuilder.get_production_wizard_task().ctx['checkout_arcadia_from_url']
        base = Arcadia.parse_url(arc).path
        arc = Arcadia.replace(arc, path=os.path.join(base, 'search', 'begemot', 'data'))
        return Arcadia.get_arcadia_src_dir(arc)

    def _merge_legacy_fresh(self, arc_root_url, arcadia_dirs_to_checkout, arcadia_path, working_copy_fresh, fresh_url_parsed):
        emanager = ExceptionManager()
        legacy_fresh_results = {}
        threads = [
            threading.Thread(target=emanager.run_target, args=(
                self.checkout_arcadia, arc_root_url, arcadia_dirs_to_checkout, arcadia_path
            )),
            threading.Thread(target=emanager.run_target, args=(self.checkout_legacy_runtime, legacy_fresh_results)),
        ]
        for thread in threads:
            thread.start()
        for thread in threads:
            thread.join()
        legacy_fresh_path = legacy_fresh_results.get('path', None)
        emanager.check_exceptions()

        runtime_legacy_url = utils.get_or_default(self.ctx, RuntimeLegacy)
        ignore_sb_conflicts = utils.get_or_default(self.ctx, IgnoreSandboxConflicts)
        do_commit = utils.get_or_default(self.ctx, DoCommit)
        verbose = utils.get_or_default(self.ctx, VerboseLogs)
        email = merge.FromLegacy(self, legacy_fresh_path, runtime_legacy_url, working_copy_fresh,
                                 os.path.join(arcadia_path, 'ya'), fresh_url_parsed.subpath,
                                 ignore_sb_conflicts, do_commit, verbose).go()
        if email:
            channel.sandbox.send_email(**email)

        if not do_commit:
            self.info += textwrap.dedent('''
                Merge is set to true, commit is set to False.
                The resources are not uploaded to sandbox, the patch generates invalid ya.make-files, build was cancelled.
            ''')
            return False

        if not self.build_required:
            self.info += '\nDoBuild set to False, exiting.'
            return False
        return True

    def on_execute(self):
        if self.ctx.get(RuntimeDataSource.name) == self.ctx.get(RuntimeLegacy.name):
            raise errors.SandboxTaskFailureError('"%s" == "%s", task arguments are invalid.' % (RuntimeDataSource.description, RuntimeLegacy.description))

        if not self.ctx.get(RobotsRevision.name):
            self.ctx[RobotsRevision.name] = int(Arcadia.info(RuntimeDataSource.Robots)['commit_revision'])
        if not self.ctx.get(ArcadiaRevision.name) or self.ctx.get(ArcadiaRevision.name + '_auto', False):
            # We want to reautodetect the revision to be able to successfully pass merge the second time after a restart
            self.ctx[ArcadiaRevision.name] = int(Arcadia.info(RuntimeDataSource.Arcadia)['commit_revision'])
            self.ctx[ArcadiaRevision.name + '_auto'] = True

        begemot_shard_builders = self._make_begemot_shard_builders()

        arcadia_path = self.abs_path('arcadia')
        arcadia_dirs_to_checkout = ['build']
        shard_paths = [b.path for b in begemot_shard_builders]
        for path in [arcadia_path] + shard_paths:
            paths.make_folder(path, delete_content=True)

        if shard_paths:
            if utils.get_or_default(self.ctx, BuildForProduction):
                self.begemot_shards_root = self._checkout_arcadia_branch()
            else:
                begemot_shards_path = os.path.join('search', 'begemot', 'data')
                arcadia_dirs_to_checkout.append(begemot_shards_path)
                self.begemot_shards_root = os.path.join(arcadia_path, begemot_shards_path)

        fresh_url = utils.get_or_default(self.ctx, RuntimeDataSource)
        if self.info is None:
            self.info = ''
        fresh_url_parsed = Arcadia.parse_url(fresh_url)

        if fresh_url_parsed.subpath:
            arcadia_dirs_to_checkout.append(fresh_url_parsed.subpath)
            # (sandboxsdk.svn) URL "arcadia:/arc/branches/begemot/stable-195/arcadia/search/wizard/data/fresh/"
            # parsed into ArcadiaURL(path=u'arc/branches/begemot/stable-195/arcadia/search/wizard/data/fresh/',
            #                        revision=None, branch=u'begemot/stable-195', tag=None, trunk=False,
            #                        subpath=u'search/wizard/data/fresh/')
            arc_root_url = fresh_url_parsed.path.replace(fresh_url_parsed.subpath, '')
            arc_root_url = Arcadia.replace(fresh_url, path=arc_root_url)
        else:
            raise ValueError('Fresh url cannot be arcadia root: %s' % fresh_url)

        self.merge_required = utils.get_or_default(self.ctx, DoMerge)
        self.build_required = utils.get_or_default(self.ctx, DoBuild)

        working_copy_fresh = os.path.join(arcadia_path, fresh_url_parsed.subpath)

        if self.merge_required:
            if not self._merge_legacy_fresh(arc_root_url, arcadia_dirs_to_checkout, arcadia_path, working_copy_fresh, fresh_url_parsed):
                return
        else:
            self.checkout_arcadia(arc_root_url, arcadia_dirs_to_checkout, arcadia_path)

        self.apply_arcadia_patch(working_copy_fresh)

        arcadia_build_root = self.abs_path('build_py_src')
        env = os.environ.copy()
        env['SANDBOX_TASK'] = str(self.id)
        env['ARCADIA_REVISION'] = str(self.ctx[ArcadiaRevision.name])
        env['ROBOTS_REVISION'] = str(self.ctx[RobotsRevision.name])
        Arcadia._switch_to_ro(arcadia_path)
        arc_sdk.do_build(
            constants.YA_MAKE_FORCE_BUILD_SYSTEM,  # self.ctx[BuildSystem.name],  # SPI-5746
            arcadia_path,
            targets=[fresh_url_parsed.subpath],
            clear_build=False,
            results_dir=arcadia_build_root,
            checkout=True,
            env=env,
        )
        pre_build_path = os.path.join(arcadia_build_root, fresh_url_parsed.subpath, 'package')
        self.apply_patch(arcadia_build_root)

        for b in begemot_shard_builders:
            b.set_static_data_dir(os.path.join(self.begemot_shards_root, b.service))
        self.validate_build_results(working_copy_fresh, pre_build_path)
        self.make_runtime_packages(pre_build_path, begemot_shard_builders)

        for b in begemot_shard_builders:
            self.update_cypress_shard(b.path, b.service, is_fresh=True, retries=5)

        parent_resources = self.ctx.get(ParentResources.name) or {}
        for res_type, res_id in parent_resources.items():
            have = channel.sandbox.list_resources(resource_type=res_type, task_id=self.id)
            if have:
                self.save_parent_task_resource(have[0].path, int(res_id), save_basename=True)
            else:
                self.info += 'Warning: parent task requested %s, but no such resource generated' % res_type


__Task__ = WizardRuntimeBuild
