# -*- coding: utf-8 -*-

from __future__ import print_function

import json
import logging
import hashlib
import os
import shutil
import subprocess
import threading
import textwrap
import traceback

from sandbox.projects import resource_types
from sandbox.common.types import resource as ctr
from sandbox.common.utils import get_task_link
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import ssh

from sandbox.projects.WizardRuntimeBuild.ya_make import YaMake

YaMake = YaMake.YaMake

logger = logging.getLogger(__name__)
debug = logger.debug


class FromLegacy:
    '''
    This class reads '.merged_revision' from the 'new' data directory (aka target_path),
    scans legacy_path for revisions newer than that and applies all incoming revisions
    file-by-file. All conflicting files are skipped and logged, all the rest changes are applied.

    SVN add, delete and modify actions are supported. Text files are patched, binary files
    are uploaded to sandbox. Conflicts for sandboxed files are checked by comparing md5sums
    for the base files, so this scheme is fragile, no patches are possible.

    Usage:
    email = merge.FromLegacy(self, legacy_fresh_path, runtime_legacy_url, working_copy_fresh,
                             os.path.join(arcadia_path, 'ya'), ignore_sb_conflicts, do_commit).go()
    if email:
        channel.sandbox.send_email(**email)  # sends notice on conflicts

    The 'go' method also sends discouraging emails suggesting commits to target_path directly
    to the authors of all processed commits (apart from generating the conflicts email).
    '''

    VAULT_RECORD = 'Begemot Commit Token'
    VAULT_OWNER = 'BEGEMOT'
    COMMITER = 'robot-begemot'
    INTREE_FILESIZE_LIMIT_BYTES = 1 * 1024 * 1024  # 1 MB

    def __init__(self, task, legacy_path, legacy_url, target_path, ya, peerdir_prefix, ignore_sb_conflicts=False, do_commit=True, verbose=False):
        '''
        :param task: sandbox task object
        :param legacy_path: working copy path of /robots/trunk/wizard-data
        :param lagacy_url: url, remote origin for legacy_path
        :param target_path: working copy path of /arc/trunk/arcadia/search/wizard/data/fresh
        :param peerdir_prefix: subpath of target_path which should be prepended to all peerdirs
            peerdirs will be formed as os.path.join(peerdir_prefix, path relative to target_path)
        :param ignore_sb_conflicts: always update all sandbox resources, discarding what's present in Arcadia
        :param verbose: produce very long merge.log for debugging
        '''
        self.generated_patch = task.create_resource(
            '/robots -> /arc backport patch',
            'fresh_from_legacy.patch',
            resource_types.WIZDATA_PATCH
        )

        with open(self.generated_patch.path, 'w') as p:
            p.write('Nothing to merge')

        self.log = task.create_resource(
            'Merge /robots -> /arc logs',
            'merge.log',
            resource_types.TASK_CUSTOM_LOGS
        )

        logger.addHandler(logging.FileHandler(self.log.path))
        self.legacy_path = legacy_path
        self.legacy_url = legacy_url
        self.target_path = target_path
        self.ya = ya
        self.last_merged_rev_file = os.path.join(target_path, '.merged_revision')
        self.task = task
        self.ignore_sb_conflicts = ignore_sb_conflicts
        self.do_commit = do_commit
        self.verbose = verbose

        # Map: filename -> file revision to upload
        self._binary_files = {}

        self.legacy_url_parsed = Arcadia.parse_url(legacy_url)
        self.subpath = self.legacy_url_parsed.subpath
        if self.subpath is None and self.legacy_url_parsed.path.startswith('robots/'):
            self.subpath = self.legacy_url_parsed.path[len('robots/'):]
        self.legacy_prefix_to_strip = '/' + self.subpath.strip('/') + '/'
        self.peerdir_prefix = peerdir_prefix
        self._deleted_resources = []

    def _get_sandbox_id(self, filename, _SANDBOX_IDS={}):
        '''
        Read yamake_filename, find FROM_SANDBOX(12345 OUT filename) and return the resource id
        :param filename: local file name (with legacy_prefix_to_strip stripped)
        :return: int, id. Return None if nothing found.
        '''
        if filename not in _SANDBOX_IDS:
            if os.path.exists(os.path.join(self.target_path, os.path.dirname(filename), 'ya.make')):
                yamake = YaMake(os.path.join(self.target_path, os.path.dirname(filename), 'ya.make'))
                sandboxed = yamake.get_sandboxed(os.path.basename(filename))
                _SANDBOX_IDS[filename] = sandboxed.resource
            else:
                _SANDBOX_IDS[filename] = None

        return _SANDBOX_IDS[filename]

    def _create_yamake_if_missing(self, yamakefile):
        if not os.path.exists(yamakefile):
            dirname = os.path.dirname(yamakefile)
            if not os.path.exists(dirname):
                os.makedirs(dirname)
            with open(yamakefile, 'w') as f:
                print('OWNER(g:begemot)', file=f)
                print('UNION()', file=f)
                print('FILES()', file=f)
                print('END()', file=f)

    def _add_smth_to_yamake(self, filename, do_add):
        yamakefile = os.path.join(self.target_path, os.path.dirname(filename), 'ya.make')
        self._create_yamake_if_missing(yamakefile)
        yamake = YaMake(yamakefile)
        do_add(yamake)
        with open(yamakefile, 'w') as y:
            yamake.dump(y)

    def _add_file_to_yamake(self, filename):
        '''
        Find a suitable ya.make in the target_dir for filename and add FILE(filename) there
        :return: error if any, None else
        '''
        def _do_add(yamake):
            yamake.files.add(os.path.basename(filename))
        self._add_smth_to_yamake(filename, _do_add)

    def _add_dir_to_yamake(self, filename):
        '''
        Find a suitable ya.make in the target_dir for filename and add FILE(filename) there
        :return: error if any, None else
        '''
        def _do_add(yamake):
            yamake.recurse.add(os.path.basename(filename))
            yamake.peerdir.add(os.path.join(self.peerdir_prefix, filename))
        self._add_smth_to_yamake(filename, _do_add)

    def _rm_file_from_yamake(self, filename, sandboxed):
        '''
        Find a suitable ya.make in the target_dir for filename and remove FILE(filename) from it
        :return: error if any, None else
        '''
        yamakefile = os.path.join(self.target_path, os.path.dirname(filename), 'ya.make')
        if not os.path.exists(yamakefile):
            self.task.info += 'Unable to find a suitable ya.make to delete %s. You may need to delete it manually.' % filename
            debug('Unable to delete {filename} automatically. {yamakefile} does not exist'.format(**locals()))
            return
        yamake = YaMake(yamakefile)
        debug("svn delete %s" % filename)
        filename = os.path.basename(filename)
        if sandboxed:
            yamake.remove_sandbox_resource(filename)
        else:
            yamake.files.discard(filename)
        # maybe filename was a directory?
        yamake.recurse.discard(filename)
        yamake.peerdir.discard(filename)
        with open(yamakefile, 'w') as y:
            yamake.dump(y)

    def _merge_to_sandbox(self, filename, rev0, _MAY_OVERWRITE_SANDBOX={}):
        '''
        Check if filename should be placed to sandbox.
        Compare sandboxed reource md5 from arcadia vs filename@rev0 from /robots.

        :param _MAY_OVERWRITE_SANDBOX: Map: filename -> is it ok to overwrite it
        :param filename: file path inside the repository (relative to /robots/trunk/)
        :param rev0 int: the revision where all merges started (to compare base versions vs arcadia-sandboxed)
        :return: no_conflict (bool)
        '''
        info = 'sandboxing %s... ' % filename
        proceed = self.ignore_sb_conflicts or _MAY_OVERWRITE_SANDBOX.get(filename, True)
        if not self.ignore_sb_conflicts and filename not in _MAY_OVERWRITE_SANDBOX:
            resource = self._get_sandbox_id(filename)
            if not resource:
                # No previous version stored in Arcadia, no conflicts possible
                _MAY_OVERWRITE_SANDBOX[filename] = True
                debug(info + ' (a new file)')
                return True

            m = hashlib.md5()
            url = Arcadia.replace(Arcadia.append(self.legacy_url, filename), revision=rev0)
            temp_filename = self.task.abs_path('data_for_md5')
            Arcadia.export(url, temp_filename)
            with open(temp_filename) as data:
                m.update(data.read())
            m = m.hexdigest()
            os.unlink(temp_filename)

            canon_md5 = channel.sandbox.get_resource(resource) if resource else None
            if canon_md5:
                canon_md5 = canon_md5.file_md5
                proceed = canon_md5 == m
            else:
                # if no previous file version found, we're safe to overwrite
                pass

            _MAY_OVERWRITE_SANDBOX[filename] = proceed

        info += ['conflict', 'ok'][proceed]
        debug(info)
        return proceed

    def _merge_to_arcadia(self, filename, rev):
        '''
        Apply patch for filename@rev to the arcadia working copy

        :param filename: relative filename inside /robots/trunk
        :return: no_conflict (bool)
        '''
        legacy_file_path = os.path.join(self.legacy_path, filename)
        strip = self.legacy_path.count('/') + 1
        PIPE = subprocess.PIPE
        patch_fname = os.path.realpath('patch')  # currently no two patches are generated in parallel
        with open(patch_fname, 'w') as p:
            p.write(Arcadia.diff(legacy_file_path, change=rev))
        patch = subprocess.Popen([self.ya, 'tool', 'svn', 'patch', patch_fname, '--strip', str(strip)], cwd=self.target_path, stdin=PIPE, stdout=PIPE, stderr=PIPE)
        stdout, stderr = patch.communicate(None)
        retcode = patch.poll()
        assert retcode is not None
        success = retcode == 0
        if not success:
            rejects = ''
            try:
                rejfname = '%s.rej' % os.path.join(self.target_path, filename)
                with open(rejfname) as r:
                    for i, line in enumerate(r):
                        rejects += line
                        if i == 50:
                            rejects += '<file truncated, only 50 first lines shown>\n'
                            break
                self.log = self.task.create_resource(
                    'Failed merge for %s' % filename,
                    rejfname,
                    resource_types.TASK_CUSTOM_LOGS
                )
            except Exception as x:
                rejects = str(x)

            debug(textwrap.dedent(
                '''
                Failed patching {filename}@{rev}:
                ---8<------8<---
                exit code {retcode}
                stdout:
                {stdout}
                ---
                stderr:
                {stderr}
                ---
                rejects:
                {rejects}
                ---8<------8<---
                '''
            ).format(**locals()))
        debug('patching %s... %s' % (filename, ['conflict', 'ok'][success]))
        return success

    def _process_revision(self, rev0, details):
        '''
        :param rev0: last merged revision to detect conflicts in sandboxed resources
        :param details: Arcadia revision info
        :return: the list of skipped files
        '''
        rev = details['revision']
        debug('Processing r%s (by %s)...\n---\n%s\n---' % (rev, details['author'], details['msg']))
        conflicts = []
        l = len(self.legacy_prefix_to_strip)
        for action, path in details['paths']:
            if self.verbose:
                debug('  process_revision: action = %s, path = %s' % (action, path))
            assert path.startswith(self.legacy_prefix_to_strip)
            filename = path[l:]
            if action not in 'MAD':
                debug('Only Append, Modify and Delete SVN actions are supported.\n ... Skipped action: %s %s@%s' % (action, filename, rev))
                conflicts.append(filename)
                continue

            source_file = os.path.join(self.legacy_path, filename)
            target_file = os.path.join(self.target_path, filename)
            try:
                is_sandboxed = bool(self._get_sandbox_id(filename))
            except Exception:
                is_sandboxed = False
            exists = os.path.exists(target_file)

            if action == 'D':
                self._rm_file_from_yamake(filename, is_sandboxed)
                self._binary_files.pop(filename, None)
                if exists:
                    Arcadia.delete(target_file)
                continue

            if not os.path.exists(source_file):
                debug('%s: ignored, missing in robots/, deleted by now' % filename)
                continue

            if os.path.isdir(source_file):
                if not os.path.exists(target_file):
                    os.makedirs(target_file)
                    Arcadia.add(target_file)
                self._add_dir_to_yamake(filename)
                continue

            is_small = os.stat(source_file).st_size <= FromLegacy.INTREE_FILESIZE_LIMIT_BYTES
            is_binary = filename.rsplit('.', 1)[-1] in ['trie', 'bin', 'data']  # todo: ask svn properly for svn:mime-type

            if exists:
                # Patching an existing file
                success = self._merge_to_arcadia(filename, rev)
                if not success:
                    conflicts.append(filename)
                continue

            if not is_sandboxed and not is_binary and is_small:
                # SVN add, new file, not sandboxed

                if self.verbose:
                    debug('  is_sandboxed: {is_sandboxed}, is_binary: {is_binary}, is_small: {is_small}'.format(**locals()))
                    debug('  file size: %s' % os.stat(source_file).st_size)

                dirname = os.path.dirname(target_file)
                if not os.path.exists(dirname):
                    os.makedirs(dirname)  # TODO: use python3 and exists_ok=True

                shutil.copy(source_file, target_file)
                debug("svn add %s" % filename)
                error = self._add_file_to_yamake(filename)
                Arcadia.add(target_file, parents=True)
                if error:
                    conflicts.append(error)
                continue

            # Adding a file to sandbox
            success = self._merge_to_sandbox(filename, {'A': rev, 'M': rev0}[action])
            if success:
                self._binary_files[filename] = rev
            else:
                conflicts.append(filename)

        debug('=======================')
        return conflicts

    def _generate_email(self, to, body):
        to = list(to)
        subj = '[wizard-fresh] Runtime merge failed'
        body = textwrap.dedent(
            '''
            Hello, committer to /robots/wizard-data!

            Failed auto-merging one of your commits from {url} to Arcadia.

            Conflicting data will not be processed automatically,
            you need to manually merge it to Arcadia.

            Merge task: {task_url}

            {details}
            ---
            You may find some help from these people:
            https://abc.yandex-team.ru/services/search-wizard/

            See also:
            https://a.yandex-team.ru/arc/trunk/arcadia/search/wizard/data/fresh/Readme.md
            https://wiki.yandex-team.ru/begemot/fresh/#avtomjorzhilka

            Virtually yours, Yandex Begemot
            '''
        ).format(
            url=self.legacy_url,
            task_url=get_task_link(self.task.id),
            details=body
        )

        debug("Generated email:\n======\nTo: %s\nSubject: %s\n----%s\n======\n" % (
            to, subj, body))

        return {
            'mail_to': to,
            'mail_cc': '',
            'mail_subject': subj,
            'mail_body': body
        }

    def _sandbox_upload(self, file_path, filename_for_comment):
        filename = os.path.basename(file_path)
        dirname = os.path.dirname(file_path)
        for attempt in xrange(10):
            try:
                argv = [
                    self.ya,
                    'upload', filename,
                    '--type=WIZARD_DATA',
                    '--description',
                    '%s<br />\nWizard fresh auto-port (<a href="http://st/REQWIZARD-965">REQWIZARD-965</a>)' % filename_for_comment,
                    '--user', FromLegacy.COMMITER,
                    '--owner', 'SEARCH-RELEASERS',
                    '--json-output',
                    '--http',
                ]
                ya = subprocess.Popen(argv, cwd=dirname, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                stdout, stderr = None, None
                stdout, stderr = ya.communicate()
                if ya.returncode != 0:
                    raise subprocess.CalledProcessError(ya.returncode, argv, stdout, stderr)
                ya_result = json.loads(stdout)
                resource_id = ya_result.get('resource_id')
                if not resource_id or ya_result.get('task', {}).get('status', 'SUCCESS') not in ['SUCCESS', 'FINISHING']:
                    logging.debug('Uploading failed: %s' % json.dumps(ya_result, indent=1))
                    continue
                channel.sandbox.set_resource_attribute(resource_id, 'ttl', 'inf')
                return int(resource_id)
            except subprocess.CalledProcessError as x:
                logging.error('Failed to upload "%s" to sandbox:\n%s\n%s' % (filename_for_comment, x.output, x.stderr))
            except Exception as exception:
                logging.error(textwrap.dedent(
                    '''
                    Failed to upload "{filename_for_comment}" to sandbox (Exception):
                    {exception}
                    -- stdout --
                    {stdout}
                    -- stderr --
                    {stderr}
                    -- -- --
                    '''.format(**locals())
                ))
        return None

        # Use this code back as soon as ya make is able to rename OUT files FROM_SANDBOX:
        # https://ml.yandex-team.ru/thread/devtools/2370000004736131023/
        resource = self.task.create_resource(
            description='%s<br />\nWizard fresh auto-port (<a href="http://st/REQWIZARD-965">REQWIZARD-965</a>)' % filename_for_comment,
            resource_path=file_path,
            resource_type=resource_types.WIZARD_DATA,
            attributes={'ttl': 'inf'}
        )
        self.task.mark_resource_ready(resource)
        resource.restart_policy = ctr.RestartPolicy.IGNORE
        return resource.id

    def _update_sandbox_resource_id(self, filename, new_resource_id):
        yamakefile = os.path.join(self.target_path, os.path.dirname(filename), 'ya.make')
        self._create_yamake_if_missing(yamakefile)
        yamake = YaMake(yamakefile)
        yamake.update_sandbox_resource(os.path.basename(filename), new_resource_id, compression='FILE')
        with open(yamakefile, 'w') as y:
            yamake.dump(y)

    def _sandbox_upload_thread(self, file_path, filename_for_comment, newres):
        try:
            if self.do_commit:
                new_resource = self._sandbox_upload(file_path, filename_for_comment)
            else:
                new_resource = 0
            newres[filename_for_comment] = new_resource
        except Exception:
            logging.error('Sandbox upload failed for %s:\n%s' % (filename_for_comment, traceback.format_exc()))
            newres[filename_for_comment] = None

    def _delete_from_sandbox(self, resource, off_the_record=False):
        try:
            channel.sandbox.drop_resource_attribute(resource, 'ttl')
            if not off_the_record:
                self._deleted_resources.append(resource)
        except Exception as x:
            self.skipped.append('Failed to delete the resource %s:\n... %s' % (resource, x))

    def _undelete_all_from_sandbox(self):
        for r in self._deleted_resources:
            try:
                debug('sandbox undelete %s' % r)
                channel.sandbox.set_resource_attribute(r, 'ttl', 'inf')
            except Exception as x:
                debug("Failed to undelete the resource %s: %s" % (r, x))

    def _drop_all_resources(self, new_resources, off_the_record=False):
        for key, resource in new_resources.iteritems():
            if resource is not None:
                debug('sandbox delete %s' % resource)
                self._delete_from_sandbox(resource, off_the_record)
            else:
                debug('skip sandbox delete for %s (because resource is %s)' % (key, resource))

    def _upload_all_resources_to_sandbox(self):
        to_upload = {}
        for fname, r in self._binary_files.iteritems():
            if r not in to_upload:
                to_upload[r] = [fname]
            else:
                to_upload[r].append(fname)

        threads = {}
        new_resources = {}
        for r, fnames in to_upload.iteritems():
            Arcadia.update(self.legacy_path, revision=r)
            for fname in fnames:
                exported_fname = self.task.abs_path(fname)
                exported_dir = os.path.dirname(exported_fname)
                if not os.path.exists(exported_dir):
                    os.makedirs(exported_dir)
                Arcadia.export(os.path.join(self.legacy_path, fname), exported_fname)
                fname_for_comment = '%s@%s' % (os.path.join(self.legacy_url, fname), r)
                thread = threading.Thread(target=self._sandbox_upload_thread, args=(exported_fname, fname_for_comment, new_resources))
                assert fname_for_comment not in threads, 'A thread already exists for %s, threads:\n%s' % (fname_for_comment, json.dumps(threads, indent=2))
                threads[fname_for_comment] = thread
                thread.start()

        for r, fnames in to_upload.iteritems():
            for fname in fnames:
                fname_for_comment = '%s@%s' % (os.path.join(self.legacy_url, fname), r)
                assert fname_for_comment in threads, 'No threads found for %s, threads:\n%s' % (fname_for_comment, json.dumps(threads, indent=2))
                logging.debug('[_finalize_commit] waiting for the file "%s"...' % fname_for_comment)
                threads[fname_for_comment].join()
                logging.debug('[_finalize_commit] ... joined "%s"' % fname_for_comment)
                assert fname_for_comment in new_resources, 'No resource found for %s, resources:\n%s' % (fname_for_comment, json.dumps(new_resources, indent=2))
                new_resource = new_resources[fname_for_comment]
                debug_note = ' (disabled, no_commit)' if not self.do_commit else ''
                debug('sandbox upload %s -> %s%s' % (fname_for_comment, new_resource, debug_note))
                if new_resource is None:
                    self.skipped.append('r%s: %s (ya upload failed 10 times in a row)' % (r, fname))
                    self._undelete_all_from_sandbox()
                    self._drop_all_resources(new_resources, off_the_record=True)
                    # Avoid bad half-merged commits
                    raise ValueError('Ya upload is broken. Try again later.')

                resource = self._get_sandbox_id(fname)
                if resource:
                    if self.do_commit:
                        debug('sandbox delete %s' % resource)
                        self._delete_from_sandbox(resource)
                    else:
                        debug('sandbox delete %s (not really, no_commit, disabled)' % resource)
                self._update_sandbox_resource_id(fname, new_resource)
        return new_resources

    def _finalize_commit(self):
        '''
        Upload all binaries, generate the final patch and commit to Arcadia
        '''
        if self.do_commit:
            with ssh.Key(self, key_owner=FromLegacy.VAULT_OWNER, key_name=FromLegacy.VAULT_RECORD):
                new_resources = self._upload_all_resources_to_sandbox()
        else:
            new_resources = self._upload_all_resources_to_sandbox()

        with open(self.last_merged_rev_file, 'w') as rev_f:
            rev_f.write(str(self.revisions[-1]['revision']))

        with open(self.generated_patch.path, 'w') as p:
            p.write(Arcadia.diff(self.target_path))
        self.task.mark_resource_ready(self.generated_patch.id)

        self.task.info += 'Merging <a href="%s">the patch</a>\n' % channel.sandbox.get_resource(self.generated_patch.id).proxy_url
        commit_message = '[WizardRuntimeBuild] Applied commits from %s\n\n%s\n%s\nSKIP_CHECK' % (
            self.legacy_url,
            ' '.join(['r%s' % r['revision'] for r in self.revisions]),
            get_task_link(self.task.id)
        )
        self.task.info += '============\nCommit message:\n%s\n============\n' % commit_message
        debug('=======================\nCommitting to Arcadia as %s:\n---\n%s\n=======================' % (FromLegacy.COMMITER, commit_message))
        if self.do_commit:
            with ssh.Key(self, key_owner=FromLegacy.VAULT_OWNER, key_name=FromLegacy.VAULT_RECORD):
                try:
                    response = Arcadia.commit(self.target_path, commit_message, user=FromLegacy.COMMITER)
                except Exception as e:
                    debug('Commit failed, restoring sandbox resources\n%s' % e)
                    # Drop all resources that have not actually been committed to ya.make
                    self._undelete_all_from_sandbox()
                    self._drop_all_resources(new_resources, off_the_record=True)
                    raise
        else:
            response = 'Commits disabled (debugging)\n'
        debug(response)
        self.task.info += '%s\n' % (['', ''] + response.split('\n'))[-2]

    def _send_deprecation_email(self, authors):
        to = list(authors)
        subj = '[wizard-fresh] Your commit has been merged to Arcadia'
        body = textwrap.dedent(
            '''
            Hello, committer to /robots/wizard-data!

            Your recent commit has been merged to Arcadia,
            and from now on you should commit directly there.
            Production reads Arcadia only, and more and more people
            use arcadia/search/wizard/data/fresh instead of /robots/wizard-data.

            Merge task: {task_url}
            ---
            See also:
            https://wiki.yandex-team.ru/begemot/fresh/#jarobotchtomnedelat

            Virtually yours, Yandex Begemot
            '''
        ).format(
            task_url=get_task_link(self.task.id),
        )

        channel.sandbox.send_email(**{
            'mail_to': to,
            'mail_cc': ['gluk47'],
            'mail_subject': subj,
            'mail_body': body
        })

    def go(self):
        '''
        Merge all new revisions from legacy_path to target_path.
        The last revision processed is written to target_path/.merged_revision
        If there are conflicts during merge, they are recorded, and emails are sent to commit authors and to begemot owners,
        and the conflicting revisions are ignored.
        '''
        email_to = set(['gluk47', 'vladmiron', 'entity-search-notifications'])
        try:
            with open(self.last_merged_rev_file) as rev_f:
                rev0 = int(rev_f.read())
        except Exception as x:
            raise ValueError('Target_path is invalid, failed to read %s/%s: %s' % (self.target_path, '.merged_revision', str(x)))

        rev1 = int(Arcadia.info(self.legacy_path)['commit_revision'])
        if rev0 >= rev1:
            return None
        self.revisions = '%s:%s' % (rev0 + 1, rev1)
        debug('Merging revisions %s...' % self.revisions)
        email = ''
        self.skipped = []
        merged_commits_authors = set()
        abort_commit = False

        try:
            self.revisions = Arcadia.log(self.legacy_path, rev0 + 1, rev1)
            debug('Found revisions', ' '.join([str(r['revision']) for r in self.revisions]))
            for rev in self.revisions:
                conflicts = self._process_revision(rev0, rev)
                if conflicts:
                    if self.do_commit:
                        email_to.add(rev['author'])  # though most likely we'll try to email to some robot
                    self.skipped += ['r%s: %s' % (rev['revision'], f) for f in conflicts]
                else:
                    merged_commits_authors.add(rev['author'])
        except Exception as x:
            if not self.do_commit:
                raise  # we're debugging, no point in trying to commit what's possible
            self.task.info += 'failed,\n%s\n' % x
            email += traceback.format_exc() + '\n'
            abort_commit = True

        if abort_commit:
            email += 'No changes have been committed to Arcadia, the task will be restarted.\n\n'
        else:
            try:
                self._finalize_commit()
            except Exception as x:
                self.task.info += 'failed,\n%s\n' % x
                email += traceback.format_exc() + '\n'

        if self.skipped:
            str_skipped = '\n'.join(self.skipped)
            email += '\nThe following files were not merged because of conflicts:\n%s' % str_skipped
            self.task.info += 'Some files were skipped while merging:\n%s\n\nAn email has been sent to %s\n' % (str_skipped, ', '.join(email_to))

        if email:
            email = self._generate_email(email_to, email)
        else:
            email = None
        if self.do_commit:
            self._send_deprecation_email(merged_commits_authors)
        self.task.mark_resource_ready(self.log.id)
        return email
