#!/usr/bin/env python3
# -*- coding:utf-8 -*-

'''
This is an unofficial adapter for ya make, created by gluk47@
to support easy addition of new files to Arcadia from scripts.

See class YaMake and the end of the script for the usage hints.
'''

from __future__ import print_function

from collections import defaultdict
from .lexer import keywords
from .parser import Module, FromSandbox
import string  # for python2 maketrans


class YaMake:
    '''
    Usage:
        yamake = YaMake()
        yamake.parse(filename)

    all macros are lowercased and imported in the object as sets
        yamake.owner.add('my-robot')
        yamake.files.remove('Readme.md')
        yamake.peerdir.remove('/')
        yamake.update_sandbox_resource('data.bin', id=4444, compression='FILE')
        yamake.kind = 'UNION'

    The result can be dumped as either json or ya.make:
        yamake.dump(sys.stdout) #  as ya.make
        print(yamake.dumps_json(indent=2))
    '''

    def __init__(self, filename=None):
        self.module = None
        self.from_sandbox = {}
        self.owner = set()
        if filename:
            self.parse(filename)

    @staticmethod
    def create(kind, owners):
        if isinstance(owners, str):
            owners = [owners]
        self = YaMake()
        self.parse_data('''
        OWNER({owners})
        {kind}()
        END()
        '''.format(kind=kind, owners=' '.join(owners)))
        return self

    def parse_data(self, data, fname=None):
        self.module = Module.parse_data(data, fname)

        for key, value in self.module.__dict__.items():
            self.__dict__[key.lower()] = value

    def parse(self, fname):
        with open(fname) as f:
            self.parse_data(f.read(), fname)

    def dump(self, file_obj):
        formatted = YaMake._preformat_dict(self.__dict__)

        if 'condition' in self.__dict__:
            formatted['condition'] = 'IF(%s)\n' % ' '.join(self.condition)
            formatted['endif'] = 'ENDIF()\n'

        if self.kind:
            formatted['kind'] = '\n%s(%s)\n' % (self.kind, ' '.join(self.header_args) or '')
        file_obj.write(YaMake._format_as_ymake(formatted))

    def dumps_json(self, indent=2):
        return self.module.dumps(indent) if self.module else 'null'

    def get_sandboxed(self, filenames):
        '''
        Get information about a FROM_SANDBOX records for the given filenames
        :param filenames: a tuple (several filenames) or a string (one filename).
            The selected record will have exactly this tuple (set and order) of files
        :return: the relevant record or a dummy empty record with id=0
        '''
        if isinstance(filenames, str):
            filenames = (filenames,)
        if filenames in self.from_sandbox:
            return self.from_sandbox[filenames]
        return FromSandbox(resource=0, filenames=filenames)

    def update_sandbox_resource(self, filenames, id, compression):
        if isinstance(filenames, str):
            filenames = (filenames,)
        if filenames in self.from_sandbox:
            fs = self.from_sandbox[filenames]
            fs.resource = id
            fs.compression = compression
        else:
            self.from_sandbox[filenames] = FromSandbox(id, filenames, compression)

    def remove_sandbox_resource(self, filenames):
        if isinstance(filenames, str):
            filenames = (filenames,)
        self.from_sandbox.pop(filenames, None)

    @staticmethod
    def skip_comments(lst):
        for l in lst:
            if l and not l.startswith('#'):
                yield l

    def __getattr__(self, key):
        for words in ['INMODULE', 'OUTMODULE', 'ANYWHERE']:
            if key.upper() in keywords['%s_OPERATOR_WORDS' % words]:
                self.__dict__[key] = set()
                break
        return self.__dict__[key]

    @staticmethod
    def _normalize_string_for_sorting(s):
        try:
            return s.lstrip('#').lower().translate(string.maketrans('-_,:', '....'))
        except Exception:
            # python3:
            return s.lstrip('#').lower().translate({ord(x): '.' for x in '-_,:'})

    @staticmethod
    def _preformat_dict(src):
        formatted = defaultdict(str)
        src_from_sandbox = src.get('from_sandbox', src.get('FROM_SANDBOX', None))
        for k, v in src.items():
            k = k.lower()
            if k == 'comments' and v:
                formatted['comments'] = '\n'.join(v) + '\n'
                continue
            if not v:
                continue
            if k in ['condition', 'conditional_inmodule', 'kind', 'from_sandbox']:
                continue

            if isinstance(v, list):
                for program in v:
                    text_value = ''
                    line_len = 0
                    option_value_len = 0
                    endl_inside_value = False
                    operator = False
                    for i, w in enumerate(program):
                        line_len += len(w) + 1
                        if not option_value_len and not operator:
                            if w.startswith('-') and i + 1 < len(program) and not program[i + 1].startswith('-'):
                                option_value_len = len(' ') + len(program[i + 1])
                            long_line = line_len + option_value_len > 50
                            operator = w[0] not in '-$' and w == w.upper() and not w.isdigit()

                            if text_value and (long_line or operator):
                                text_value += '\n   '
                                endl_inside_value = True
                                if not operator and long_line:
                                    text_value += '    '
                                line_len = 0
                        else:
                            option_value_len = 0
                            operator = False
                        if text_value and text_value != '    ':
                            text_value += ' '
                        text_value += w
                    brace_endl, indent = ('\n', '    ') if endl_inside_value or len(k) > 4 else ('', '')
                    formatted[k] += '{key}({endl}{indent}{value}{endl})\n'.format(
                        endl=brace_endl,
                        indent=indent,
                        key=k.upper(),
                        value=text_value
                    )
            elif isinstance(v, set):
                text_value = '\n    '.join(sorted(map(str, v), key=YaMake._normalize_string_for_sorting))
                brace_endl, indent = ('\n', '    ') if len(v) > 1 or len(k) > 4 else ('', '')
                formatted[k] = '{key}({endl}{indent}{value}{endl})\n'.format(
                    endl=brace_endl,
                    indent=indent,
                    key=k.upper(),
                    value=text_value
                )
            else:
                formatted[k] = '%s\n' % v

        if 'conditional' in src:
            for key, conditionals in src['conditional'].items():
                YaMake._format_conditionals(key, conditionals, formatted)
        if src_from_sandbox:
            from_sandbox = []
            for fs in sorted(src_from_sandbox.values(), key=lambda x: YaMake._normalize_string_for_sorting(' '.join(x.filenames))):
                from_sandbox.append('FROM_SANDBOX(%s)' % fs)
                formatted['from_sandbox'] = '\n'.join(from_sandbox) + '\n'
        return formatted

    @staticmethod
    def _format_conditionals(key, conditionals, formatted):
        key = 'cond_%s' % key
        formatted[key] = ''
        for c in conditionals:
            formatted_body = YaMake._format_as_ymake(YaMake._preformat_dict(c['body']))
            formatted[key] += '\nIF(%s)\n%s' % (
                ' '.join(c['condition']),
                '\n'.join(['    ' * (bool(l)) + l for l in formatted_body.split('\n')])
            )
            if 'else' in c:
                formatted_body = YaMake._format_as_ymake(YaMake._preformat_dict(c['else']))
                formatted[key] += 'ELSE()\n' + '\n'.join(['    ' * (bool(l)) + l for l in formatted_body.split('\n')])
            formatted[key] += 'ENDIF()\n'

    @staticmethod
    def _format_as_ymake(preformatted_dict):
        if 'kind' in preformatted_dict:
            preformatted_dict['END'] = '\nEND()\n'
        # we want owner and license to appear first always
        preamble_ops = ''.join('{0[%s]}' % op.lower() for op in keywords['OUTMODULE_OPERATOR_WORDS'] +
                                                                keywords['ANYWHERE_OPERATOR_WORDS'] +
                                                                keywords['WORD_OR_ANYWHERE_OPERATOR'] if op not in ('OWNER', 'LICENSE'))
        inmodule_ops = ''.join('{0[%s]}' % op.lower() for op in keywords['INMODULE_OPERATOR_WORDS'] +
                                                                keywords['WORD_OR_INMODULE_OPERATOR'])
        return ('{0[condition]}{0[comments]}{0[owner]}{0[license]}%s{0[cond_preamble]}' % preamble_ops +
                '{0[kind]}%s{0[from_sandbox]}{0[cond_inmodule]}' % inmodule_ops +
                '{0[END]}{0[endif]}').format(preformatted_dict).lstrip('\n')


if __name__ == '__main__':
    # Sample
    import sys

    yamake = YaMake()
    for fname in sys.argv[1:]:
        print('=== %s ===' % fname, file=sys.stderr)
        yamake.parse(fname)
        # print(yamake.dumps_json(indent=2) + "\n\n")  # JSON, if you wish
        yamake.update_sandbox_resource('newfile', id=404, compression='ZIP')
        yamake.update_sandbox_resource('newfile2', id=503, compression='FILE2')
        first_file = next(iter(yamake.from_sandbox.keys()))
        second_file = next(iter(yamake.from_sandbox.keys()))
        yamake.update_sandbox_resource(first_file, id=777777777, compression='FILE')
        yamake.remove_sandbox_resource(second_file)
        # yamake.remove_sandbox_resource('video.gzt')
        yamake.recurse.discard('server')
        yamake.recurse.discard('rules')
        yamake.files.add('added-file-123')
        print("--- Peerdirs ---\n%s\n--- --- ---\n" % '\n'.join(yamake.peerdir))
        yamake.dump(sys.stdout)
        sys.stdout.flush()
