#!/usr/bin/env python
# -*- coding:utf-8 -*-

'''
This is an unofficial parser for ya make, created by gluk47@
to support easy addition of new files to Arcadia from scripts.

See YaMake.py for detailed usage.
'''

from __future__ import print_function

import json
import logging
import ply.yacc as yacc
import sys
from .lexer import tokens, lexer  # noqa  , required by yacc
from collections import defaultdict

logging.basicConfig()
logger = logging.getLogger(__name__)
# For the following operators the values will be preserved as lists,
# and not squashed into one big set (like it happens for PEERDIR etc)
_OPERATORS_WITH_LIST_VALUES = ['RUN_PROGRAM', 'SET', 'SET_APPEND', 'PYTHON', 'COPY', 'TOOL']


def debug(x):
    return
    print(x, file=sys.stderr)


class ObjEncoder(json.JSONEncoder):
    def _tuples_to_str(self, d):
        if isinstance(d, list):
            for i, v in enumerate(d):
                d[i] = self._tuples_to_str(v)
            return d
        elif not isinstance(d, dict):
            return d

        if not d or not isinstance(next(iter(d.keys())), tuple):
            ret = d
        else:
            ret = {' '.join(k): v for k, v in d.items()}

        for k, v in ret.items():
            ret[k] = self._tuples_to_str(v)
        return ret

    def encode(self, obj):
        return json.JSONEncoder.encode(self, self._tuples_to_str(obj))

    def default(self, o):
        if isinstance(o, set):
            return list(o)
        return self.encode(o.__dict__)


class Module:
    def __init__(self, kind, body, header_args=None):
        self.__dict__ = body
        self.kind = kind
        self.header_args = header_args

    def dumps(self, indent=2):
        return json.dumps(self.__dict__, indent=indent, cls=ObjEncoder)

    @staticmethod
    def parse_data(data, fname='<preloaded data>'):
        parser = yacc.yacc()
        lexer.lineno = 1
        logger.debug('Parsing %s...' % fname)
        try:
            module = parser.parse(data)
            if module is None:
                raise ValueError('Failed parsing "%s", no module can be parsed at all (see the logs for syntax errors)' % fname)
        except Exception:
            logger.error('An error encountered while parsing "%s"' % fname)
            logger.error('The full file to parse:\n------ 8< --- 8< --- 8< ------\n%s\n------ 8< --- 8< --- 8< ------\n' % data)
            raise
        return module


class FromSandbox:
    def __init__(self, resource, filenames, compression=None, out='OUT', rename=None):
        self.resource = int(resource)
        self.filenames = filenames
        self.out = out
        self.compression = compression
        self.rename = rename

    def __hash__(self):
        return self.resource

    def __str__(self):
        compression = self.__dict__.get('compression') or ''
        if compression:
            compression += ' '
        rename = self.__dict__.get('rename') or ''
        if rename:
            rename = 'RENAME %s ' % rename
        return '{compression}{resource} {rename}{out} {filenames}'.format(
            compression=compression,
            resource=self.resource,
            rename=rename,
            out=self.out,
            filenames=' '.join(self.filenames),
        )

    def __or__(self, rhs):
        '''If we are here, we want to "merge" ( |= ) two from_sandbox for the same filenames'''
        raise ValueError('Duplicate FROM_SANDBOX for filenames {n}: resources {r1} and {r2}'.format(
            n=list(self.filenames), r1=self.resource, r2=rhs.resource)
        )


def merge_dicts(dest, src):
    for key, value in src.items():
        if key not in dest:
            dest[key] = value
            continue
        if isinstance(value, dict):
            merge_dicts(dest[key], value)
        elif isinstance(value, list):
            dest[key] += value
        else:
            dest[key] |= value


def p_module(p):
    '''module : preamble MODULE_TYPE BEGIN_BLOCK words END_BLOCK module_body END braces preamble
              | preamble WORD_OR_MODULE_TYPE BEGIN_BLOCK words END_BLOCK module_body END braces preamble'''
    result = p[6]
    debug('module: %s' % result)
    for preamble in [p[1], p[9]]:
        if preamble:
            merge_dicts(result, preamble)
    p[0] = Module(p[2], result, header_args=p[4])


def p_module_no_body(p):
    '''module : preamble'''
    debug('module no body')
    p[0] = Module(None, p[1])


def p_module_if(p):
    '''module : IF BEGIN_BLOCK words END_BLOCK module ENDIF braces '''
    debug('module if')
    module = p[5]
    module.condition = p[3]
    p[0] = module


# TODO generalize and fix this
def p_module_if_prefixed(p):
    '''module : ANYWHERE_OPERATOR_WORDS BEGIN_BLOCK words END_BLOCK IF BEGIN_BLOCK words END_BLOCK module ENDIF braces '''
    debug('module if')
    module = p[9]
    module.condition = p[7]
    p[0] = module


def p_braces(p):
    '''braces : BEGIN_BLOCK END_BLOCK'''


def p_preamble(p):
    '''preamble :
                | OUTMODULE_OPERATOR_WORDS  BEGIN_BLOCK words END_BLOCK preamble
                | WORD_OR_ANYWHERE_OPERATOR BEGIN_BLOCK words END_BLOCK preamble
                | ANYWHERE_OPERATOR_WORDS   BEGIN_BLOCK words END_BLOCK preamble'''
    debug('preamble: len(p) = %s' % len(p))
    if len(p) > 1:
        debug('... op: %s, words: %s' % (p[1], p[3]))
    if len(p) > 5:
        key = p[1]
        if key in _OPERATORS_WITH_LIST_VALUES:
            p[0] = {key: [p[3]]}
        else:
            p[0] = {key: set(p[3])}
        merge_dicts(p[0], p[5])
    else:
        p[0] = {}


def p_preamble_comment(p):
    '''preamble : COMMENT preamble'''
    p[0] = {'comments': set([p[1]])}
    merge_dicts(p[0], p[2])


def p_preamble_if(p):
    '''preamble : IF BEGIN_BLOCK words END_BLOCK preamble ENDIF braces preamble'''
    debug('preamble if(%s)' % ' '.join(p[3]))
    p[0] = {'conditional': {'preamble': [{'condition': p[3], 'body': p[5]}]}}
    merge_dicts(p[0], p[8])


def p_preamble_if_else(p):
    '''preamble : IF BEGIN_BLOCK words END_BLOCK preamble ELSE braces preamble ENDIF braces preamble'''
    debug('preamble if-else')
    p[0] = {'conditional': {'preamble': [{'condition': p[3], 'body': p[5], 'else': p[8]}]}}
    merge_dicts(p[0], p[10])


def p_module_body(p):
    '''module_body :
                   | operator module_body'''
    p[0] = defaultdict(list)
    if len(p) > 1:
        debug('module_body: %s' % p[1])
        key = list(p[1].keys())[0]
        if key in _OPERATORS_WITH_LIST_VALUES:
            p[0][key].append(p[1][key])
        else:
            merge_dicts(p[0], p[1])
        merge_dicts(p[0], p[2])


def p_operator(p):
    '''operator : INMODULE_OPERATOR_WORDS   BEGIN_BLOCK words END_BLOCK
                | ANYWHERE_OPERATOR_WORDS   BEGIN_BLOCK words END_BLOCK
                | WORD_OR_INMODULE_OPERATOR BEGIN_BLOCK words END_BLOCK
                | WORD_OR_ANYWHERE_OPERATOR BEGIN_BLOCK words END_BLOCK
                | FROM_SANDBOX              BEGIN_BLOCK from_sandbox END_BLOCK'''
    debug('operator %s: %s' % (p[1], p[3]))
    if isinstance(p[3], dict) or p[1] in _OPERATORS_WITH_LIST_VALUES:
        p[0] = {p[1]: p[3]}
    else:
        p[0] = {p[1]: set(p[3])}


def p_operator_comment(p):
    '''operator : COMMENT'''
    p[0] = {'comments': set([p[1]])}


def p_cond_operators(p):
    '''operator : IF BEGIN_BLOCK words END_BLOCK module_body ENDIF braces'''
    debug('cond operators if(%s)' % ' '.join(p[3]))
    p[0] = {'conditional': {'inmodule': [{'condition': p[3], 'body': p[5]}]}}
    debug(p[0])


def p_cond_operators_else(p):
    '''operator : IF BEGIN_BLOCK words END_BLOCK module_body ELSE braces module_body ENDIF braces'''
    debug('cond operators if-else(%s)' % ' '.join(p[3]))
    p[0] = {'conditional': {'inmodule': [{'condition': p[3], 'body': p[5], 'else': p[8]}]}}
    debug(p[0])


def p_from_sandbox_file(p):
    '''from_sandbox_file :
                         | FILE'''
    if len(p) > 1:
        p[0] = p[1]
    else:
        p[0] = None


def p_from_sandbox(p):
    '''from_sandbox : from_sandbox_file WORD OUT words
                    | from_sandbox_file WORD RENAME WORD OUT words'''
    if len(p) > 5:
        files = tuple(p[6])
        rename = p[4]
        out = p[5]
    else:
        files = tuple(p[4])
        rename = None
        out = p[3]

    value = FromSandbox(p[2], files, compression=p[1], rename=rename, out=out)
    p[0] = {files: value}


def p_words(p):
    '''words :
             | WORD words
             | COMMENT words
             | WORD_OR_INMODULE_OPERATOR words
             | WORD_OR_ANYWHERE_OPERATOR words
             | WORD_OR_MODULE_TYPE words
             | OUT words '''
    if len(p) > 1:
        p[0] = [p[1]] + p[2]
    else:
        p[0] = []


def p_error(p):
    logger.error('[YaMakeParser] syntax error: line {line}, unexpected token «{text}» ({kind})'.format(
        line=p.lineno, text=p.value, kind=p.type
    ))
    raise Exception()


if __name__ == '__main__':
    if False:
        print(tokens)
    parser = yacc.yacc()
    ret = parser.parse('''
        OWNER(
            g:wizard
            gluk47
        )
        SET(set1 key1 value1)
        RECURSE(
            replace_queries_tries_fix
        )
        UNION()
        PEERDIR(
            search/wizard/data/fresh/video/replace_queries_tries_fix
        )
        FROM_SANDBOX(FILE 369682845 OUT besttitlecrc.trie)
        FROM_SANDBOX(372246889 OUT full_series_structs.trie)
        # some comment
        # FROM_SANDBOX(unused)
        FILES(
            __init__.py
            videosyn.gzt
            # oldfile
        )
        SET(PROG1
            prog1
                --arg1
        )
        RUN_PROGRAM(
            prog1 in1.txt
                  -o out1.txt
            IN
                in1.txt
                in2.txt
            OUT out1.txt
            CWD ${ARCADIA_ROOT}
        )
        RUN_PROGRAM(p2 -o p2_out.txt
            OUT_NOAUTO p2_out
        )

        END()
    ''')

    if ret:
        print(ret.dumps())
