import os
import logging

from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters as sp

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import profiling as search_profiling
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search.components.apphost import AppHost
from sandbox.projects.common.wizard import parameters as wp
from sandbox.projects.common.wizard import utils as wizard_utils
from sandbox.projects.BegemotTestPerformance import __Task__ as bgperf


class ApphostProtobuf(sp.SandboxBoolParameter):
    name = 'apphost_binary'
    description = 'AppHost binary protocol'


class WizardTestPerformance(
    search_performance.OldShootingTask,
    task.SandboxTask,
    search_profiling.ProfilingTask
):
    type = 'WIZARD_TEST_PERFORMANCE'
    required_ram = 100 * 1024
    execution_space = 60 * 1024
    client_tags = wizard_utils.BEGEMOT_PERF_HARDWARE_TAGS & wizard_utils.WIZARD_SANDBOX_HOSTS_TAGS

    input_parameters = wp.WIZ_PERF_PARAMS + (ApphostProtobuf,) + \
        search_performance.OldShootingTask.shoot_input_parameters + \
        search_profiling.ProfilingTask.input_parameters

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        wizard_utils.setup_hosts_sdk1(self, additional_restrictions=wizard_utils.BEGEMOT_PERF_HARDWARE_TAGS)
        wizard_utils.on_enqueue(self)

    def on_execute(self):
        wizard_binary_path = self.sync_resource(self.ctx[wp.Binary.name])
        wizard_config_path = self.sync_resource(self.ctx[wp.Config.name])
        wizard_data_path = self.sync_resource(self.ctx[wp.Shard.name])
        runtime_data_path = self.sync_resource(self.ctx[wp.RuntimeData.name])

        wizard = search_components.get_wizard(
            wizard_binary_path,
            wizard_config_path,
            wizard_data_path,
            runtime_data_path,
            cache=self.ctx[wp.CacheEnable.name]
        )

        if self.ctx.get(wp.UseAppHost.name):
            apphost_path = bgperf._unpack_apphost(self.sync_resource(apihelpers.get_last_released_resource(resource_types.APP_HOST_BUNDLE).id), self.abs_path())
            self._create_apphost_graph(apphost_path, int(wizard.get_port()) + 1)
            apphost_config_path = bgperf._create_apphost_config(apphost_path)
            apphost = AppHost(self, os.path.join(apphost_path, 'app_host'), apphost_config_path)

        self._profiling_init(wizard, self.__get_perf_data_path())
        self._init_virtualenv()
        with wizard:
            if self.ctx.get(wp.UseAppHost.name):
                with apphost:
                    self._old_shoot(apphost, self.ctx[wp.Plan.name])
            else:
                self._old_shoot(wizard, self.ctx[wp.Plan.name])
            try:
                self.ctx['wizard_stat'] = wizard.get_stat().encode('utf-8')
                self.ctx['wizard_yasm_stat'] = wizard.get_yasm_stat()
            except Exception as e:
                logging.info("Can't get wizard_stat: %s", str(e))
        self._profiling_report(wizard, self.__get_perf_data_path())

    def _create_apphost_graph(self, apphost_path, port):
        graph = {
            'name': 'begemot',
            'graph': {'SERVICE': ['INIT'], 'RESPONSE': ['SERVICE']},
            'version': 2,
            'sources': {'SERVICE': {'backend_descrs': [{'addr': 'netliba://localhost:{}'.format(port)}], 'timeout': 150}},
        }
        for source in graph['sources'].values():
            source['backend_config'] = {'backend_descrs': source['backend_descrs']}
        if self.ctx.get(ApphostProtobuf.name):
            graph['sources']['CONFIG'] = {'embed': [{'type': 'begemot_config', 'binary': True, 'result_type': 'wizard'}]}
            graph['graph']['SERVICE'].append('CONFIG')
        os.mkdir(os.path.join(apphost_path, 'graphs'))
        fu.json_dump(os.path.join(apphost_path, 'graphs', 'begemot.json'), graph)

    def get_short_task_result(self):
        if self.is_completed() and "max_rps" in self.ctx:
            return "{:0.2f}".format(self.ctx["max_rps"])

    def __get_perf_data_path(self, recreate=False):
        return self.abs_path("perf.data")


__Task__ = WizardTestPerformance
