#!/usr/bin/env python
# -*- coding: utf-8 -*-
u"""Выгрузка запросов за n-дней из нано-логов."""

import argparse
import yt.wrapper as ytw

from datetime import date, timedelta

from helper import set_expiration_time

from nile.files import LocalFile
from nile.api.v1 import (
    clusters,
    Record,
    aggregators as na,
)


def arg_parser():
    u"""Собираем параметры для запуска программы."""
    parser = argparse.ArgumentParser(
        description='Собираем информацию о требуемой статистике',
        prog='wizard_stat',
        add_help=False,
    )

    parser.add_argument(
        '-h', '--help',
        action='help',
        help='Показать зто описание аргументов программы'
    )

    parser.add_argument(
        '-v', '--version',
        action='version',
        version='%(prog)s 03.2017',
        help='Показать версию программы (месяц.год)'
    )

    parser.add_argument(
        '-s', '--session-table',
        dest='session_dates',
        help='Смотреть по дневным нано-логам, дата yyyy-mm-dd \n \
            //user_sessions/pub/nano_sessions/daily',
        nargs='?',
        const=last_date(),
        default=last_date(),
        # const=last_session('user_sessions'),
        # default=last_session('user_sessions'),
    )

    parser.add_argument(
        '-o', '--postfix',
        dest='postfix',
        default='',
        help='Результирующая таблица (постфикс), необязательно',
    )

    arguments = parser.parse_args()

    return arguments


def last_date():
    u"""Вернуть вчерашнее число как строку год-месяц-день."""
    return (date.today() - timedelta(1)).strftime('%Y-%m-%d')


def parse_nano(records):
    u"""Разбор записей лога нано-сессий.
    Args:
        records: итератор по записям YT.
    Returns:
        Record: запись таблицы YT.
    """
    for record in records:
        key = record.key
        value = eval(record.value)
        ui_dict = {
            0: 'desktop',
            1: 'tablet',
            2: 'mobile',
            3: 'touch',
            4: 'app'
        }

        if key[0] != 'y':
            continue
        if value['type'] != 'REQUEST':
            continue

        query = value['query']

        if not query:
            continue

        region = value['user_region']
        ui = ui_dict[value['ui']]
        domain = value['service_dom_region']

        yield Record(query=query, domain=domain, region=region, platform=ui)


def check_q_exists(_date):
    u"""Проверить существование предыдущей выгруженной таблицы."""
    return ytw.exists('//home/antiwizard/query_viewer/queries/' + _date + '/queries')


def check_q_empty(_date):
    u"""Проверить, что выгруженная таблица пустая."""
    table = ytw.TablePath('//home/antiwizard/query_viewer/queries/' + _date + '/queries')

    return ytw.is_empty(table)


def check_nano_us_exists(_date):
    u"""Проверить существование таблицы в сессиях."""
    return ytw.exists('//user_sessions/pub/nano_sessions/daily/' + _date + '/web/clean')


def check_dates(_dates):
    u"""Проверить существование таблиц по датам, вернуть только существующие и не выгруженные."""
    existing_dates = []

    for d in _dates.split(','):
        if check_nano_us_exists(d):
            existing_dates.append(d)

    for xd in existing_dates:
        if check_q_exists(xd) and not check_q_empty(xd):
            existing_dates.remove(xd)

    return ','.join(existing_dates)


def run_nano_parse(_args):
    u"""Функция запуска разбора нано-логов."""
    files = map(LocalFile, ['query.py'])
    cluster = clusters.Hahn(pool='sup').env(
        templates=dict(
            home='//home/antiwizard/query_viewer/queries',
            sess='//user_sessions/pub/nano_sessions/daily',
        ),
        files=files
    )

    postfix = '_' + _args.postfix if _args.postfix else ''
    output_date = date.today().strftime('%Y-%m-%d') if len(_args.session_dates.split(',')) > 1 else _args.session_dates
    job = cluster.job().env(templates=dict(
        sess_dates='{' + check_dates(_args.session_dates) + '}',
        sess_clean='web/clean',
        output_date=output_date,
        queries='queries' + postfix,
        aggregated_queries='aggregated_queries' + postfix,
    ))

    input_session_table = job.table('$sess/@sess_dates/$sess_clean')

    queries = input_session_table \
        .map(
            parse_nano,
            memory_limit=16384,
            intensity='data'
        ) \
        .put('$home/$output_date/$queries')

    queries \
        .groupby('platform', 'domain', 'query') \
        .aggregate(count=na.count()) \
        .put('$home/$output_date/$aggregated_queries')

    job.run()

    templates = job.environment.templates

    set_expiration_time('{home}/{date}/{queries}'.format(
        home=templates['home'],
        date=templates['output_date'],
        queries=templates['queries'],
    ), delay=86400)

    set_expiration_time('{home}/{date}/{aggr_queries}'.format(
        home=templates['home'],
        date=templates['output_date'],
        aggr_queries=templates['aggregated_queries'],
    ), delay=86400)
