#!/usr/bin/env python
# -*- coding: utf-8 -*-

from nile.api.v1 import (
    clusters,
    Record,
    aggregators as na,
    Template,
)
from nile.files import LocalFile, RemoteFile

from datetime import date, timedelta

import argparse
import time
import yt.wrapper as ytw


def arg_parser():
    # Собираем параметры для запуска программы

    parser = argparse.ArgumentParser(
        description='Собираем информацию о требуемой статистике',
        prog='wizard_stat',
        add_help=False,
    )

    parser.add_argument(
        '-h', '--help',
        action='help',
        help='Показать зто описание аргументов программы'
    )

    table_choice = parser.add_mutually_exclusive_group()

    table_choice.add_argument(
        '-s', '--session-table',
        dest='session',
        help='Смотреть по дневным логам, дата yyyy-mm-dd \n user_sessions/pub/search/daily',
        nargs='?',
        const=last_date(),
        default=last_date(),
        # const=last_session('user_sessions'),
        # default=last_session('user_sessions'),
    )

    table_choice.add_argument(
        '-f', '--fast-session',
        dest='fast',
        help='Смотреть по быстрым (30 минут) сессиям, дата unix timestamp. \n user_sessions/pub/search/fast',
        # nargs='?',
        # const=last_session('fast_logs/user_sessions'),
    )

    parser.add_argument(
        '-o', '--postfix',
        dest='postfix',
        default='',
        help='Результирующая таблица (постфикс), необязательно',
    )

    parser.add_argument(
        '-g', '--glue',
        dest='glue',
        action='store_true',
    )

    arguments = parser.parse_args()

    return arguments


def last_date():
    return (date.today() - timedelta(1)).strftime('%Y-%m-%d')


def platform_detect(req):
    # Определение платформы по типу запроса

    platform = ''
    if req.IsA('TDesktopUIProperties'):
        platform = 'desktop'
    if req.IsA('TTouchUIProperties'):
        platform = 'touch'
    if req.IsA('TPadUIProperties'):
        platform = 'tablet'
    if req.IsA('TMobileUIProperties'):
        platform = 'mobile'
    if req.IsA('TMobileAppUIProperties'):
        platform = 'app'
    # if req.IsA('TSiteSearchUIProperties'):
    # 	platform = 'site_search'
    return platform


def cut_path(path, _bs_blocks):
    if not path:
        return 'empty'
    if path.split('/')[1] == 'parallel':
        n = 5
    else:
        n = 3

    short_path = '/' + '/'.join(path.split('/')[1:n])
    shortest_path = ''

    for bs_block in _bs_blocks:
        if short_path in bs_block.Path:
            if shortest_path:
                if len(bs_block.Path) < len(shortest_path):
                    shortest_path = bs_block.Path
            else:
                shortest_path = bs_block.Path

    # поправить отсутствие родительского path для parallel (не entity_search)
    if n == 5:
        shortest_path_list = shortest_path.split('/')[1:]
        if 'entity_search' not in shortest_path and len(shortest_path_list) > n - 1:
            shortest_path = '/' + '/'.join(shortest_path_list[:n - 1])

    return shortest_path or short_path


def get_type(_path, _adapters, length):
    if _path != 'empty':
        if len(_path.split('/')) < length:
            if length == 5 and 'parallel' not in _path:
                _path = '/parallel/result' + _path
            else:
                _path_len = len(_path.split('/'))
                _path += '/empty' * (length - _path_len)

        path_wiz_type = _path.split('/')[length - 1]
        path_wiz_subtype = _path.split('/')[length] if len(_path.split('/')) > length else ''
        path_wiz_sub_subtype = _path.split('/')[length + 1] if len(_path.split('/')) > length + 1 else ''
        wiz_type = _adapters[0] if _adapters else path_wiz_type

    else:
        path_wiz_type = path_wiz_subtype = path_wiz_sub_subtype = _path
        wiz_type = _adapters[0] if _adapters else _path

    return path_wiz_type, path_wiz_subtype, path_wiz_sub_subtype, wiz_type


def remove_empty_table(_date):
    table = ytw.TablePath('//home/search-research/antonka/wizard-queries/' + _date)

    if ytw.is_empty(table):
        ytw.remove(table)


def check_wq_exists(_date):
    table = ytw.TablePath('//home/search-research/antonka/wizard-queries/' + _date)

    return ytw.exists(table)


def check_wq_empty(_date):
    table = ytw.TablePath('//home/search-research/antonka/wizard-queries/' + _date)

    return ytw.is_empty(table)


def check_us_exists(_date):
    return ytw.exists('//user_sessions/pub/search/daily/' + _date + '/clean')


class ParseUsersessions:
    # Класс для работы с YT. Класс нужен был для MR, чтобы пробрасывать параметры.
    # Здесь, наверное, тоже необходимо.

    def __init__(self, opts):
        self.args = opts
        self.input = []
        self.session_type = 'fast' if self.args.fast else 'daily'
        if self.args.fast:
            self.input = self.args.fast
        else:
            self.input = self.args.session

    def __call__(self):
        self.start_yt()

    @staticmethod
    def process_session(groups, correct, error):
        """функция обработки сессии."""

        import libra

        for key, records in groups:
            uid = key.key
            if 'y' != uid[0]:
                continue
            try:
                s = libra.ParseSession(records, './blockstat.dict')
            except RuntimeError:
                continue

            for request in s:

                if not request.IsA('TWebRequestProperties'):
                    continue

                if not platform_detect(request):
                    continue
                try:
                    if request.PageNo != 0:
                        continue

                    platform = platform_detect(request)
                    domain = request.ServiceDomRegion
                    region = request.UserRegion
                    query = request.Query
                    bs_blocks = request.GetBSBlocks()

                    for block in request.GetMainBlocks():
                        block_res = block.GetMainResult()

                        if block_res.IsA('TBlenderWizardResult') or block_res.IsA('TWizardResult'):
                            path = cut_path(block_res.ConvertedPath, bs_blocks)
                            path_type, path_subtype, path_sub_subtype, wiz_type = get_type(path, block.Adapters, 3)
                            if len(path.split('/')) < 3 and path != 'empty':
                                error(Record(
                                    query=query,
                                    platform=platform,
                                    domain=domain,
                                    region=region,
                                    name=block_res.Name or 'empty',
                                    path=path,
                                    wiz_type=wiz_type,
                                    exps=', '.join([i.TestID for i in request.GetTestInfo()]),
                                    fail_reason='Insufficient number of path fields',
                                    position='left',
                                ))

                            correct(Record(
                                query=query,
                                platform=platform,
                                domain=domain,
                                region=region,
                                name=block_res.Name or 'empty',
                                path=path,
                                path_type=path_type,
                                path_subtype=path_subtype,
                                path_sub_subtype=path_sub_subtype,
                                wiz_type=wiz_type,
                                exps=', '.join([i.TestID for i in request.GetTestInfo()]),
                            ))

                    for block in request.GetParallelBlocks():
                        block_res = block.GetMainResult()

                        if block_res.IsA('TBlenderWizardResult'):
                            path = cut_path(block_res.ConvertedPath, bs_blocks)
                            path_type, path_subtype, path_sub_subtype, wiz_type = get_type(path, block.Adapters, 5)
                            if len(path.split('/')) < 5 and path != 'empty':
                                error(Record(
                                    query=query,
                                    platform=platform,
                                    domain=domain,
                                    region=region,
                                    name=block_res.Name or 'empty',
                                    path=path,
                                    wiz_type=wiz_type,
                                    exps=', '.join([i.TestID for i in request.GetTestInfo()]),
                                    fail_reason='Insufficient number of path fields',
                                    position='right',
                                ))

                            correct(Record(
                                query=query,
                                platform=platform,
                                domain=domain,
                                region=region,
                                name=block_res.Name or 'empty',
                                path=path,
                                path_type=path_type,
                                path_subtype=path_subtype,
                                path_sub_subtype=path_sub_subtype,
                                wiz_type=wiz_type,
                                exps=', '.join([i.TestID for i in request.GetTestInfo()]),
                            ))

                except AttributeError:
                    # raise ValueError(request.__dict__)
                    error(Record(
                        vals=request.__dict__,
                        yuid=request.UID,
                        reqid=request.ReqID,
                        query=request.Query
                    ))

    def reduce_and_aggr(self, _cluster, _dates, _output_date, _input_date, _postfix, _append=False):
        job = _cluster.job().env(
            templates=dict(
                sess_dates='{' + _dates + '}',
                wizard_queries=_output_date + _postfix,
                wizard_queries_aggr=_output_date + _postfix + '_aggr',
                errors=_output_date + _postfix + '_' + 'errors',
            )
        )
        session_table = job.table('$sess/$sess_type/{}/clean'.format(_input_date))
        result, errors = session_table \
            .groupby('key') \
            .sort('subkey') \
            .reduce(
                self.process_session,
                memory_limit=8192,
                intensity='data'
            )

        result.put('$home/$wizard_queries', append=_append)
        errors.put('$home/$errors', append=_append)

        # wizard_queries = job.table('$home/$wizard_queries')

        result \
            .groupby('platform', 'domain', 'path', 'wiz_type', 'query') \
            .aggregate(count=na.count()) \
            .put('$home/$wizard_queries_aggr')

        job.run()

    def start_yt(self):
        files_remote = map(RemoteFile, ['//statbox/statbox-dict-last/blockstat.dict', '//statbox/resources/libra.so'])
        files_local = map(LocalFile, ['wizard_query.py'])
        files = files_remote + files_local

        cluster = clusters.Hahn(pool='sup').env(
            templates=dict(
                home='home/search-research/antonka/wizard-queries',
                sess='user_sessions/pub/search',
                sess_type=self.session_type
            ),
            files=files
        )
        cur_date = time.strftime('%Y-%m-%d', time.localtime())

        dates = [i for i in self.input.split(',')]
        for d in dates:
            if check_wq_exists(d) and not check_wq_empty(d):
                dates.remove(d)
        dates = ','.join(dates)

        postfix = '_' + self.args.postfix if self.args.postfix else ''
        if self.args.glue:
            self.reduce_and_aggr(cluster, dates, cur_date, '@sess_dates', postfix, _append=True)
        else:
            for d in dates.split(','):
                self.reduce_and_aggr(cluster, d, d, '$sess_dates', postfix, _append=False)


def main(_args):
    ps = ParseUsersessions(_args)
    ps()

    tables = [str(x) for x in Template('{' + _args.session + '}').unfold()]
    for d in tables:
        try:
            print(d + '_errors')
            remove_empty_table(d + '_errors')
        except:
            pass
