import logging
import os
import shutil

import sandbox.common.types.task as ctt
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.vcs.git import GitLfs

from sandbox import common
import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.sdk2 import yav
import requests


class DockerContentRes(sdk2.Resource):
    """
    Directory with dockerfile.
    """
    releasable = False
    restart_policy = ctr.RestartPolicy.DELETE


class BuildDockerImageWto(sdk2.Task):
    """
    Builds docker image from git repository.
    Inspired by BUILD_MARKET_DOCKER_IMAGES, tries to be
    more generic.
    """
    checkout_path = 'src'

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        container = sdk2.parameters.Container("LXC Container", default_value=682327247, required=True)  # 29
        with sdk2.parameters.Group("Git parameters") as git_block:
            repository = sdk2.parameters.String("Repository", required=True)
            branch = sdk2.parameters.String("Branch", default="master")
            commit = sdk2.parameters.String("Commit", default=None)
            use_lfs = sdk2.parameters.Bool("Use LFS", default=False)
            filter_branches = sdk2.parameters.Bool("Filter branches", default=True)
            update_submodules = sdk2.parameters.Bool("Init and update submodules", default=False)
            path_to_dockerfile = sdk2.parameters.String(
                "Path to directory with Dockerfile inside the repository", default="."
            )
            dockerfile_name = sdk2.parameters.String(
                "Dockerfile name (defaults to 'Dockerfile')", default="Dockerfile"
            )
            ssh_vault_name = sdk2.parameters.String("Vault item with ssh key for git access")
            ssh_vault_owner = sdk2.parameters.String("Vault item owner")

        with sdk2.parameters.Group("Registry parameters") as docker_block:
            registry_tags = sdk2.parameters.List(
                "Tags to publish image with (registry.yandex.net/<this tags>)"
            )
            oauth_vault_name = sdk2.parameters.String(
                "Vault item with oauth token for "
                "registry.yandex.net (vault item name)"
            )
            oauth_vault_owner = sdk2.parameters.String("Vault item owner")
            registry_login = sdk2.parameters.String("Yandex login to use with docker login")
            cache_from_image_tag = sdk2.parameters.String(
                "--cache-from image tag (if present, will be used as docker build --cache-from=<this>)"
            )
            docker_build_args = sdk2.parameters.List("Docker --build-arg option")
            teamcity_build_number = sdk2.parameters.String("teamcity_build_number", required=True)
            app_name = sdk2.parameters.String("app_name", required=True)
            release_version = sdk2.parameters.String("release_version", required=True)
            build_configuration_id = sdk2.parameters.String("build_configuration_id", required=True)
        docker_setup_script = sdk2.parameters.Resource("Script for docker setup", ui=None)

    def git_clone(self, use_lfs, update_submodules):
        with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            if os.path.exists(self.checkout_path):
                shutil.rmtree(self.checkout_path)
            if use_lfs:
                git = GitLfs(self.Parameters.repository, filter_branches=self.Parameters.filter_branches)
            else:
                git = Git(self.Parameters.repository, filter_branches=self.Parameters.filter_branches)
            git.clone(self.checkout_path, self.Parameters.branch, self.Parameters.commit)
            if update_submodules:
                git.execute("submodule", "update", "--init", "--recursive", cwd=self.checkout_path)

    def prepare_resource(self):
        self.git_clone(self.Parameters.use_lfs, self.Parameters.update_submodules)
        dir_path = os.path.join(self.checkout_path, self.Parameters.path_to_dockerfile)
        dockerfile_path = os.path.join(dir_path, self.Parameters.dockerfile_name)
        if not os.path.exists(dockerfile_path):
            raise common.errors.TaskError("Dockerfile {} not found inside the repository".format(
                dockerfile_path)
            )
        # make sure dockerfile is named 'Dockerfile' so that the child task could fetch it
        shutil.move(dockerfile_path, os.path.join(dir_path, 'Dockerfile'))
        secret = yav.Secret("sec-01dvvg3gstjarn4qgffzcg3v7s")
        teamcity_oauth = secret.data()["TEAMCITY_OAUTH"]
        url = "https://teamcity.yandex-team.ru/app/rest/builds/buildType:(id:{}),number:{}/artifacts/content/{}.war".format(
            self.Parameters.build_configuration_id, self.Parameters.teamcity_build_number, self.Parameters.app_name)
        headers = {"Authorization": "OAuth {0}".format(teamcity_oauth)}
        filedata = requests.get(url, headers=headers)
        with open("{0}/{1}.war".format(dir_path, self.Parameters.app_name), 'wb') as f:
            f.write(filedata.content)
        resource = DockerContentRes(
            self,
            "Dockerfile and contents from {repo}".format(
                repo=self.Parameters.repository,
            ),
            dir_path
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def build_image(self, resource):
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        kwargs = {
            BuildDockerImageV6.PackagedResource.name: resource.id,
            BuildDockerImageV6.RegistryTags.name: self.Parameters.registry_tags,
            BuildDockerImageV6.RegistryLogin.name: self.Parameters.registry_login,
            BuildDockerImageV6.VaultItemName.name: self.Parameters.oauth_vault_name,
            BuildDockerImageV6.VaultItemOwner.name: self.Parameters.oauth_vault_owner,
            BuildDockerImageV6.CacheFromImageTag.name: self.Parameters.cache_from_image_tag,
            BuildDockerImageV6.DockerBuildArgs.name: self.Parameters.docker_build_args
        }
        if self.Parameters.docker_setup_script:
            kwargs[BuildDockerImageV6.DockerSetupScript.name] = self.Parameters.docker_setup_script.id

        logging.info(str(kwargs))
        sub_task = task_class(
            task_class.current,
            description="Building and publishing Docker image from Git source (by {})".format(self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        self.Context.sub_task_id = sub_task.id
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.first_step:
            resource = self.prepare_resource()
            self.build_image(resource)
        with self.memoize_stage.second_step:
            sub_task = self.find(id=self.Context.sub_task_id).first()
            if sub_task.status not in ctt.Status.Group.SUCCEED:
                raise common.errors.TaskFailure("Subtask is failed with status {}".format(sub_task.status))
