import logging

from sandbox import sdk2
from sandbox.projects.yabs.base_bin_task import BaseBinTask


class DeployToQloudWto(BaseBinTask):
    class Parameters(BaseBinTask.Parameters):
        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            release_version = sdk2.parameters.String(
                "Release version",
                default="stable"
            )
            resource_attrs = sdk2.parameters.Dict(
                "Filter resource by",
                default={"name": "DeployToQloud"}
            )

        with sdk2.parameters.Group("Deployment parameters") as params:
            qloud_component_path = sdk2.parameters.String(
                "Path to Qloud component",
                description="project.application.environment.component",
                required=True,
            )
            image_tag = sdk2.parameters.String(
                "Docker image registry tag",
                description="registry.yandex.net/my_image:latest",
                required=True,
            )
            qloud_token_vault_name = sdk2.parameters.String(
                "Vault secret with qloud oauth token",
                required=True,
            )
            comment = sdk2.parameters.String(
                "Comment for deploy",
                required=True,
            )

    @staticmethod
    def split_image_tag(image_tag):
        if ":" in image_tag:
            name, tag = image_tag.split(":")
        else:
            name, tag = image_tag, "latest"

        if name.startswith("registry.yandex.net/"):
            name = name[len("registry.yandex.net/"):]

        return name, tag

    def _find_required_component(self, env, component_name):
        for component in env.get("components", []):
            if component["componentName"] == component_name:
                return component

        msg = "No component for path {} were found".format(self.Parameters.QloudEnvironmentParameter)
        logging.error(msg)
        raise Exception(msg)

    def _release_qloud_env(self, qloud_component_path, image_tag, comment):
        from sandbox.projects.rasp.qloud.api import (
            QloudPublicApi,
            QloudPrivateApi,
            ComponentSpecification,
        )
        qloud_token = sdk2.Vault.data(
            self.owner,
            self.Parameters.qloud_token_vault_name,
        )
        qloud_api = QloudPublicApi(qloud_token)
        component_spec = ComponentSpecification.parse_from_path(qloud_component_path)
        environment = qloud_api.dump_environment(component_spec.environment_id)

        image_name, image_tag = self.split_image_tag(image_tag)
        image_hash = QloudPrivateApi(qloud_token).docker_hash(image_name, image_tag)

        logging.info(
            "Updating component with path %s, registry %s and tag %s",
            qloud_component_path,
            image_name,
            image_tag,
        )

        component = self._find_required_component(environment, component_spec.name)
        component["properties"]["hash"] = image_hash
        component["properties"]["repository"] = "registry.yandex.net/{}:{}".format(image_name, image_tag)
        environment["comment"] = comment

        qloud_api.upload_environment(environment)

    def on_execute(self):
        qloud_component_path = self.Parameters.qloud_component_path
        image_tag = self.Parameters.image_tag
        comment = self.Parameters.comment

        self._release_qloud_env(qloud_component_path, image_tag, comment)
